package com.bxm.dailyegg.task.service.impl;

import com.bxm.dailyegg.common.constant.TaskCommonConstant;
import com.bxm.dailyegg.task.cache.TaskCacheHolder;
import com.bxm.dailyegg.task.config.TaskProperties;
import com.bxm.dailyegg.task.constant.RedisCacheConstant;
import com.bxm.dailyegg.task.model.domain.UserTaskFinishStatisticsMapper;
import com.bxm.dailyegg.task.model.dto.TaskListItemDTO;
import com.bxm.dailyegg.task.model.entity.UserTaskFinishStatisticsEntity;
import com.bxm.dailyegg.task.service.UserTaskInfoService;
import com.bxm.dailyegg.user.dto.UserAccountDTO;
import com.bxm.dailyegg.user.facade.UserAccountFacadeService;
import com.bxm.dailyegg.user.param.UserAccountOpsParam;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @author lowi
 * @date 2022/2/21 17:18
 */
@Service
@Slf4j
@AllArgsConstructor
public class UserTaskInfoServiceImpl implements UserTaskInfoService {

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserTaskFinishStatisticsMapper userTaskFinishStatisticsMapper;

    private final TaskProperties taskProperties;

    private final UserAccountFacadeService userAccountFacadeService;

    private final TaskCacheHolder taskCacheHolder;

    @Override
    public Integer getUserDailyTaskFinishNum(Long userId, Long taskId) {
        Integer userTaskFinish = redisHashMapAdapter.get(builderDailyTaskKey(userId), taskId.toString(), Integer.class);
        return userTaskFinish == null || userTaskFinish == 0 ? 0 : userTaskFinish;
    }

    private KeyGenerator builderDailyTaskKey(Long userId) {
        String date = DateUtils.formatDate(new Date());
        return RedisCacheConstant.USER_DAILY_TASK_NUM.copy().appendKey(userId).appendKey(date);
    }

    private KeyGenerator builderBubbleTaskKey(Long userId) {
        String date = DateUtils.formatDate(new Date());
        return RedisCacheConstant.USER_BUBBLE_TASK_NUM.copy().appendKey(userId).appendKey(date);
    }

    private KeyGenerator builderUserDailyTaskFinishInfo(Long userId) {
        String date = DateUtils.formatDate(new Date());
        return RedisCacheConstant.USER_DAILY_TASK_FINISH_INFO.copy().appendKey(userId).appendKey(date);
    }

    @Override
    public Boolean bubbleTaskFinish(Long userId, Long taskId) {
        Integer userTaskFinish = redisHashMapAdapter.get(
            builderBubbleTaskKey(userId),
            taskId.toString(),
            Integer.class
        );
        return userTaskFinish == null || userTaskFinish == 0 ? Boolean.FALSE : Boolean.TRUE;
    }

    @Override
    public Boolean dailyTaskFinish(Long userId, Long taskId) {

        Boolean userTaskFinishInfo = redisHashMapAdapter.get(
            builderUserDailyTaskFinishInfo(userId), taskId.toString(), Boolean.class);
        return Objects.equals(userTaskFinishInfo, Boolean.TRUE);
    }

    @Override
    public void initUserTaskFinishInfo(Long userId, List<TaskListItemDTO> taskListItemDTO) {
        KeyGenerator key = builderUserDailyTaskFinishInfo(userId);
        //如果初始化信息不存在,则加载
        if (!redisHashMapAdapter.hasKey(key)) {
            Map<String, Boolean> taskMap = new HashMap<>();
            for (TaskListItemDTO listItemDTO : taskListItemDTO) {
                taskMap.put(listItemDTO.getTaskId().toString(), Boolean.FALSE);
            }
            taskMap.put(TaskCommonConstant.GUIDE_TASK_ID.toString(), taskCacheHolder.hasGuideAwardRecord(userId));
            //任务当天有效
            redisHashMapAdapter.putAll(key, taskMap);
            redisHashMapAdapter.expire(key, DateUtils.getCurSeconds());
        }
    }

    @Override
    public UserTaskFinishStatisticsEntity getUserFinishDailyTaskNum(Long userId) {
        return userTaskFinishStatisticsMapper.getUserTaskFinishInfo(userId);
    }

    @Override
    public Message sendFoodsAndLog(UserAccountOpsParam userAccountOpsParam) {
        return userAccountFacadeService.changeAccount(userAccountOpsParam);
    }

    @Override
    public void updateUserExtFinishNum(Long userId, Integer currNum) {
        if (currNum == null) {
            UserTaskFinishStatisticsEntity userTaskFinishStatisticsEntity = new UserTaskFinishStatisticsEntity();
            userTaskFinishStatisticsEntity.setCreateTime(new Date());
            userTaskFinishStatisticsEntity.setNum(1);
            userTaskFinishStatisticsEntity.setUserId(userId);
            userTaskFinishStatisticsEntity.setModifyTime(new Date());
            userTaskFinishStatisticsMapper.insert(userTaskFinishStatisticsEntity);
        } else {
            userTaskFinishStatisticsMapper.updateUserFinishTaskNum(userId, currNum);
        }
    }

    @Override
    public UserAccountDTO getUserAccountInfo(Long userId) {
        return userAccountFacadeService.loadUserAccount(userId);
    }
}
