package com.bxm.fossicker.message.mix;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.message.domain.UserDeviceMapper;
import com.bxm.fossicker.message.enums.PushStrategyEnum;
import com.bxm.fossicker.message.mix.strategy.IPushStrategy;
import com.bxm.fossicker.message.mix.strategy.impl.DefaultPushStrategy;
import com.bxm.fossicker.message.param.MixPushParam;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * 混合推荐分发器，根据消息配置的推送策略进行依次推送，直到某一种策略执行成功或全部失败
 *
 * @author liujia
 * @date 2020/03/27 15:12
 */
@Component
@Slf4j
public class MixPushDispatcher {

    private Map<PushStrategyEnum, IPushStrategy> strategyMap;

    private IPushStrategy defaultStrategy = new DefaultPushStrategy();

    private final UserDeviceMapper userDeviceMapper;

    @Autowired
    public MixPushDispatcher(List<IPushStrategy> pushStrategies, UserDeviceMapper userDeviceMapper) {
        this.userDeviceMapper = userDeviceMapper;
        strategyMap = Maps.newHashMap();

        pushStrategies.forEach(item -> strategyMap.put(item.type(), item));
    }

    /**
     * 根据最优先的推送策略进行推送，推送失败时进行降级处理，直到处理完成所有的推送类型
     *
     * @param param 推送参数
     */
    @Async
    public void push(MixPushParam param) {
        exec(param);
    }

    private Message exec(MixPushParam param) {
        if (log.isDebugEnabled()) {
            log.debug("推送参数：{}", JSON.toJSONString(param));
        }

        if (param.getStrategyList().size() == 0) {
            log.info("所有推送策略均失败，推送参数：{}", JSON.toJSONString(param));

            return Message.build(false, "所有推送方式均失败");
        }

        if (param.getUserPushInfo() == null) {
            param.setUserPushInfo(userDeviceMapper.loadUserInfo(param.getUserId()));
        }

        PushStrategyEnum firstStrategy = param.getStrategyList().get(0);
        //匹配推送策略
        Message pushResult = strategyMap.getOrDefault(firstStrategy, defaultStrategy).push(param);

        if (!pushResult.isSuccess()) {
            //当前推送策略推送失败则尝试下一种推送推送策略，直到将所有的配置推送执行完
            param.removeStrategy(firstStrategy);
            return exec(param);
        }

        return pushResult;
    }
}
