package com.bxm.fossicker.message.push.platform.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.fossicker.enums.PlatformEnum;
import com.bxm.fossicker.message.domain.MessageMapper;
import com.bxm.fossicker.message.domain.PushMessageMapper;
import com.bxm.fossicker.message.domain.PushUserMapper;
import com.bxm.fossicker.message.entity.MessageBean;
import com.bxm.fossicker.message.entity.PushUserBean;
import com.bxm.fossicker.message.entity.UserDeviceBean;
import com.bxm.fossicker.message.push.platform.IPushPlatform;
import com.bxm.fossicker.vo.PushMessage;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;

/**
 * 抽象的推送平台类
 * 处理推送相关的共同内容
 *
 * @author liujia
 */
public abstract class AbstractPushPlatform implements IPushPlatform {

    Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private PushMessageMapper pushMessageMapper;

    static final String SUCCESS = "ok";

    static final String FAILED = "failed";

    @Override
    public void push(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.info("开始推送消息，message：[{}],userSize:[{}]", message, userDevice.size());
        if (userDevice.size() > 0) {
            if (userDevice.size() == 1) {
                singlePush(message, userDevice.get(0));
            } else {
                groupPush(message, userDevice);
            }
            changePushNum(message, userDevice);
        }
        logger.info("推送消息结束，message：[{}],userSize:[{}]", message, userDevice.size());
    }

    /**
     * 推送完成后更新用户触达人数
     * 非默认（即指定单人推送或固定人数推送的情况）才需要进行人数的更新
     *
     * @param message    消息信息
     * @param userDevice 用户列表
     */
    private void changePushNum(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.info("消息[{}]更新触达人数[{}]", message.getPayloadInfo().getMsgId(), userDevice.size());
        pushMessageMapper.updateNum(message.getPayloadInfo().getMsgId(), userDevice.size());
    }

    /**
     * 推送结果回调处理
     */
    void callback(PushMessage message, UserDeviceBean user, String response) {
        callback(message, ImmutableList.of(user), response);
    }

    void callback(PushMessage message, List<UserDeviceBean> users, String response) {
        MybatisBatchBuilder.create(PushUserMapper.class, users).run((mapper, element) -> {
            PushUserBean entity = new PushUserBean();
            entity.setId(sequenceCreater.nextLongId());
            entity.setChecked(0);
            entity.setMessageId(message.getPayloadInfo().getMsgId());
            entity.setPlatform(element.getPlatform());
            entity.setUserId(element.getUserId());
            entity.setPushTime(new Date());
            entity.setPushResult(response);
            entity.setRegisterClient(element.getPhoneModel() + "");
            return mapper.insert(entity);
        });

        if (message.isPersistence()) {
            doPersistence(message, users);
        }
    }

    /**
     * 持久化用户通知表
     */
    private void doPersistence(PushMessage message, List<UserDeviceBean> users) {
        List<MessageBean> messageList = Lists.newArrayList();
        for (UserDeviceBean userDeviceBean : users) {
            MessageBean messageVo = new MessageBean();
            messageVo.setTitle(message.getTitle());
            messageVo.setContent(message.getContent());
            messageVo.setUserId(userDeviceBean.getUserId());
            messageVo.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
            messageVo.setStatus((byte) 0);
            messageVo.setLinkParam(JSON.toJSONString(message.getPayloadInfo()));
            messageList.add(messageVo);
        }

        MybatisBatchBuilder.create(MessageMapper.class, messageList).run(MessageMapper::insertSelective);
    }

    /**
     * 推送消息到群组
     *
     * @param message    消息体
     * @param userDevice 用户设备信息列表
     */
    abstract void groupPush(PushMessage message, List<UserDeviceBean> userDevice);

    /**
     * 判断用户设备是否iOS
     *
     * @param userDevice 用户设备信息
     * @return true表示用户设备为iOS
     */
    boolean isIOS(UserDeviceBean userDevice) {
        if (null == userDevice) {
            return false;
        }

        return PlatformEnum.IOS.getCode() == userDevice.getPhoneModel();
    }
}
