package com.bxm.fossicker.message.push.platform.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.fossicker.enums.PlatformTypeEnum;
import com.bxm.fossicker.enums.TemplateTypeEnum;
import com.bxm.fossicker.message.config.AppPushProperties;
import com.bxm.fossicker.message.entity.UserDeviceBean;
import com.bxm.fossicker.message.push.umeng.AndroidNotification;
import com.bxm.fossicker.message.push.umeng.UmengNotification;
import com.bxm.fossicker.message.push.umeng.android.AndroidUnicast;
import com.bxm.fossicker.message.push.umeng.ios.AlertMsg;
import com.bxm.fossicker.message.push.umeng.ios.IOSUnicast;
import com.bxm.fossicker.vo.PushMessage;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static com.bxm.fossicker.enums.PushSoundEnum.IOS_MUTE;

@Component
@RefreshScope
@Slf4j
public class UmengPushPlatform extends AbstractPushPlatform {

    private static final String HOST = "http://msg.umeng.com";

    private static final String POST_PATH = "/api/send";

    @Resource
    private HttpClientService httpClientService;

    @Resource
    private AppPushProperties appPushProperties;

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.UMENG;
    }

    @Override
    public Message singlePush(PushMessage message, UserDeviceBean userDevice) {
        return execPush(message, userDevice, 3);
    }

    private Message execPush(PushMessage message, UserDeviceBean userDevice, int maxAttempts) {
        if (maxAttempts <= 0) {
            return Message.build(false);
        }

        logger.debug("消息推送类型为：[{}]", message.getType());
        UmengNotification unicast;
        if (isIOS(userDevice)) {
            unicast = createIOSUnicast(message, userDevice);
        } else {
            unicast = createAndroidUnicast(message, userDevice);
        }
        String postUrl = HOST + POST_PATH;
        String postBody = unicast.getPostBody();
        String sign = StringUtils.EMPTY;

        logger.debug("推送消息：[{}]", postBody);

        try {
            sign = DigestUtils.md5Hex(("POST" + postUrl + postBody + unicast.getAppMasterSecret()).getBytes("utf8"));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        postUrl = postUrl + "?sign=" + sign;

        try {
            String response = httpClientService.doPost(postUrl, postBody);
            logger.debug("umeng push result:[{}]", response);

            if (null == response) {
                return execPush(message, userDevice, --maxAttempts);
            }

            String result = parseResult(response);
            if (SUCCESS.equals(result)) {
                //推送完成回调
                callback(message, userDevice, result);
                return Message.build(true);
            } else {
                logger.error("推送失败，重新推送。消息:[{}],失败原因：[{}]", message, result);
                callback(message, userDevice, FAILED);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            callback(message, userDevice, FAILED);
        }

        return Message.build(false);
    }


    private String parseResult(String response) {
        JSONObject json = JSONObject.parseObject(response);
        String ret = json.getString("ret");
        if (StringUtils.equalsIgnoreCase("SUCCESS", ret)) {
            return SUCCESS;
        } else {
            return response;
        }
    }

    @Override
    void groupPush(PushMessage message, List<UserDeviceBean> userDevice) {
        for (UserDeviceBean userDeviceBean : userDevice) {
            logger.debug("当前推送的用户信息为：[{}]", userDevice.toString());
            singlePush(message, userDeviceBean);
        }
    }

    /**
     * 创建ios单播推送消息
     */
    private IOSUnicast createIOSUnicast(PushMessage message, UserDeviceBean userDevice) {
        try {
            IOSUnicast unicast = new IOSUnicast();

            unicast.setAppMasterSecret(appPushProperties.getUmengIosAppSecret());
            unicast.setPredefinedKeyValue("appkey", appPushProperties.getUmengIosAppKey());

            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userDevice.getPushToken());
            AlertMsg alertMsg = new AlertMsg(message.getTitle(), message.getContent());
            unicast.setAlert(JSONObject.toJSON(alertMsg));

            unicast.setContentAvailable(0);
            unicast.setBadge(1);

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }

            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getIosSound());
            } else {
                unicast.setSound(message.getSound().getIosSound());
            }

            //组装payload
            unicast.setCustomizedField("extend", JSONObject.toJSON(message.getPayloadInfo().getExtend()));
            unicast.setCustomizedField("type", message.getPayloadInfo().getType());
            unicast.setCustomizedField("title", message.getTitle());
            unicast.setCustomizedField("content", message.getContent());
            if (StringUtils.isNotBlank(message.getPayloadInfo().getProtocol())) {
                unicast.setCustomizedField("protocol", message.getPayloadInfo().getProtocol());
            }
            unicast.setCustomizedField("msgId", message.getPayloadInfo().getMsgId());
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }

            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 创建安卓单播消息
     */
    private AndroidUnicast createAndroidUnicast(PushMessage message, UserDeviceBean userDevice) {
        try {
            AndroidUnicast unicast = new AndroidUnicast();

            unicast.setAppMasterSecret(appPushProperties.getUmengAndroidAppSecret());
            unicast.setPredefinedKeyValue("appkey", appPushProperties.getUmengAndroidAppKey());

            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userDevice.getPushToken());
            //设置通知内容
            unicast.setBody(message.getTitle(), message.getTitle(), message.getContent());

            //设置提醒方式
            unicast.setPlay(message.isShock(), message.isMute());
            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getAndriodSound());
            } else {
                unicast.setSound(message.getSound().getAndriodSound());
            }

            if (TemplateTypeEnum.TRANSMISSION.equals(message.getType())) {
                unicast.setDisplayType(AndroidNotification.DisplayType.MESSAGE);
            } else {
                unicast.setDisplayType(AndroidNotification.DisplayType.NOTIFICATION);
            }
            unicast.setCustomField(JSONObject.parseObject(JSONObject.toJSONString(message.getPayloadInfo())));
            //设置默认打开app
            unicast.goAppAfterOpen();

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }
            unicast.setExtra(JSONObject.parseObject(JSON.toJSONString(message.getPayloadInfo())));
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }
            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }
}
