package com.bxm.fossicker.message.sms;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.message.config.MessageProperties;
import com.bxm.fossicker.message.enums.SmsTempEnum;
import com.bxm.fossicker.message.enums.SmsTypeEnum;
import com.bxm.fossicker.message.param.SendSmsCodeParam;
import com.bxm.fossicker.message.sms.impl.AbstractSmsPlatform;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Primary;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 短信发送执行类，短信会存在多个服务商切换的情况，对其进行适配
 * 如果有需要，可以在此处实现服务降级处理
 *
 * @author liujia
 */
@Component
@Primary
@Slf4j
public class SmsSendSendAdapter implements SmsSendService {

    private final MessageProperties messageProperties;

    private Map<String, AbstractSmsPlatform> smsPlatformMap = Maps.newHashMap();

    private ApplicationContext applicationContext;

    public static final String DEFAULT = "default";

    @Autowired
    public SmsSendSendAdapter(MessageProperties messageProperties) {
        this.messageProperties = messageProperties;
    }

    @Override
    public Message sendSmsCode(String phone, SmsTempEnum smsTemp) {
        return this.sendSmsCode(SendSmsCodeParam.builder()
                .phone(phone)
                .smsTemp(smsTemp)
                .build());
    }

    @Override
    public Message sendSmsCode(SendSmsCodeParam param) {
        log.debug("sendSmsCode:[{}],smsPlatformMap:[{}]",JSON.toJSONString(param), JSON.toJSONString(smsPlatformMap));
        AbstractSmsPlatform platform = getSmsPlatform(param.getSmsTemp());
        log.debug("sendSmsCode - platform:[{}]",JSON.toJSONString(platform));
        Message message = platform.sendSmsCode(param);
        if (!message.isSuccess()) {
            log.warn("发送短信失败: {}, 请求参数: {}", message.getLastMessage(), JSON.toJSONString(param));
        }
        return message;
    }

    @Override
    public Message sendVariableSms(String phone, SmsTempEnum smsTemp, Object... args) {
        AbstractSmsPlatform platform = getSmsPlatform(smsTemp);
        return platform.sendVariableSms(phone, smsTemp, args);
    }

    @Override
    public Message multiSend(List<String> phoneList, String content) {
        AbstractSmsPlatform platform = smsPlatformMap.get(messageProperties.getActivedSmsPlatform());
        return platform.multiSend(phoneList, content);
    }

    /**
     * 根据短信类型，获取发送平台
     * <p>
     * 253用来发送营销短信
     * submail发送验证码
     *
     * @param smsType
     * @return
     * @author Gonzo
     */
    private AbstractSmsPlatform getSmsPlatform(SmsTempEnum smsType) {

        // 如果配置了默认平台
        if (DEFAULT.equals(messageProperties.getActivedSmsPlatform())) {
            log.debug("getSmsPlatform - 默认平台");
            return smsPlatformMap.get(messageProperties.getActivedSmsPlatform());
        }

        // 验证码
        if (SmsTypeEnum.SMS_CODE.equals(smsType.getSmsType())) {
            log.debug("getSmsPlatform - 验证码");
            return smsPlatformMap.get("submail");
        }

        // 自定义参数的短信（营销短信）
        if (SmsTypeEnum.ARGUMENT_FROM_CONFIG.equals(smsType.getSmsType())
                || SmsTypeEnum.ARGUMENT.equals(smsType.getSmsType())) {
            log.debug("getSmsPlatform - 自定义参数的短信（营销短信）");
            return smsPlatformMap.get("253");
        }

        // 默认返回默认的短信平台
        return smsPlatformMap.get(DEFAULT);
    }


    @EventListener(ContextRefreshedEvent.class)
    public void setApplicationContext(ContextRefreshedEvent event) throws BeansException {
        Map<String, AbstractSmsPlatform> platformMap = event.getApplicationContext().getBeansOfType(AbstractSmsPlatform.class);
        Collection<AbstractSmsPlatform> handlers = platformMap.values();
        handlers.forEach(platform -> smsPlatformMap.put(platform.name(), platform));
    }
}
