package com.bxm.fossicker.message.sms.impl;

import com.bxm.fossicker.base.facade.model.EquipmentDTO;
import com.bxm.fossicker.base.facade.service.EquipmentFacadeService;
import com.bxm.fossicker.message.config.MessageProperties;
import com.bxm.fossicker.message.constant.MsgRedisKey;
import com.bxm.fossicker.message.domain.MsgSmsMapper;
import com.bxm.fossicker.message.entity.SmsBean;
import com.bxm.fossicker.message.enums.SmsTempEnum;
import com.bxm.fossicker.message.enums.SmsTypeEnum;
import com.bxm.fossicker.message.param.SendSmsCodeParam;
import com.bxm.fossicker.message.sms.SmsSendService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.Validater;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.AsyncTaskExecutor;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.bxm.fossicker.message.constant.MsgRedisKey.STR_MOBILE_MAX_SMS;
import static com.bxm.fossicker.message.constant.MsgRedisKey.STR_TEMP_SMS_CODE;

/**
 * 抽象的短信发送平台
 *
 * @author liujia
 * <p>
 * 253用来发送营销短信
 * submail发送验证码
 */
@Slf4j
public abstract class AbstractSmsPlatform implements SmsSendService {

    @Autowired
    protected RedisStringAdapter redisStringAdapter;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    protected MessageProperties messageProperties;

    @Autowired
    protected MsgSmsMapper msgSmsMapper;

    @Autowired
    private AsyncTaskExecutor asyncTaskExecutor;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private EquipmentFacadeService equipmentFacadeService;

    public static final String NONE_DEIVCE_ID = "0";

    /**
     * 平台名称
     *
     * @return 平台名称用于和配置项匹配，决定使用哪个短信平台进行推送
     */
    public abstract String name();

    @Override
    public Message sendVariableSms(String phone, SmsTempEnum smsTemp, Object... args) {
        // 验证类型
        if (null == smsTemp) {
            return Message.build(false).setMessage("短信内容不存在");
        }

        // 验证手机号码
        if (!Validater.checkPhone(phone)) {
            return Message.build(false).setMessage("不是一个有效的手机号码");
        }

        String smsTempStr = getSmsTemp(smsTemp);

        if (StringUtils.isBlank(smsTempStr)) {
            return Message.build(false).setMessage("不支持的短信模板");
        }

        // 发送短信
        asyncTaskExecutor.submit(() -> doSendVariableSms(phone, smsTempStr, args));

        return Message.build();
    }

    /**
     * 获取短信模板
     *
     * @param smsType
     * @return
     */
    private String getSmsTemp(SmsTempEnum smsType) {
        String temp = null;

        // 获取短信模板
        if (SmsTypeEnum.ARGUMENT_FROM_CONFIG.equals(smsType.getSmsType())) {
            // 从配置中获取
            temp = messageProperties.getSmsTemp().get(smsType.getTemplate());

        } else if (SmsTypeEnum.ARGUMENT.equals(smsType.getSmsType())) {
            temp = smsType.getTemplate();
        }

        return temp;
    }

    @Override
    public Message sendSmsCode(String phone, SmsTempEnum smsTemp) {
        return this.sendSmsCode(SendSmsCodeParam.builder()
                .phone(phone)
                .smsTemp(smsTemp)
                .build());
    }

    @Override
    public Message sendSmsCode(SendSmsCodeParam param) {
        // 零时, 增加接口调用次数统计
        incrementTempTotalTimes(param);


        String phone = param.getPhone();
        SmsTempEnum smsTemp = param.getSmsTemp();

        // 校验基本信息
        Message message = defaultCheck(param);
        if (!message.isSuccess()) {
            return message;
        }

        // 是否是指定的手机号
        boolean isSpecificPhone = isSpecificPhone(phone);
        KeyGenerator key = STR_TEMP_SMS_CODE.copy().appendKey(phone).appendKey(smsTemp.name());

        // 验证发送限额
        if (!isSpecificPhone) {
            message = checkSendLimit(key, param);
            if (!message.isSuccess()) {
                return message;
            }
        }

        // 生成验证码
        String code = isSpecificPhone ? "1234" : RandomUtils.getShortCode(4);
        String content = smsTemp.getTemplate().replace("{}", code);

        redisStringAdapter.set(key, code, messageProperties.getValidCodeExpiredSecond());

        if (!isSpecificPhone) {
            asyncTaskExecutor.submit(() -> {
                doSendSmsCode(phone, content);
                // 增加发送次数
                incrementTimes(param);
                incrementTempSuccessTimes(param);
            });
        }

        return Message.build();
    }

    private Message defaultCheck(SendSmsCodeParam param) {
        String phone = param.getPhone();
        SmsTempEnum smsTemp = param.getSmsTemp();

        //验证类型
        if (null == smsTemp) {
            return Message.build(false).setMessage("短信内容不存在");
        }

        // 只支持验证码发送
        if (!SmsTypeEnum.SMS_CODE.equals(smsTemp.getSmsType())) {
            return Message.build(false).setMessage("不支持的短信类型");
        }

        //验证手机号码
        if (!Validater.checkPhone(phone)) {
            return Message.build(false).setMessage("不是一个有效的手机号码");
        }

        return Message.build();
    }

    /**
     * 发送次数限制
     *
     * @param key
     * @param param
     * @return
     */
    private Message checkSendLimit(KeyGenerator key, SendSmsCodeParam param) {
        // ip 限制
        if (StringUtils.isBlank(param.getIpAddr())) {
            // 如果拿不到ip 则直接不发送
            log.error("无法获取当前请求ip，限制发送验证码短信,发送参数：{}", param);
            return Message.build(false).setMessage("短信发送失败");
        } else {
            // 否则判断是否超过限制
            KeyGenerator ipAddrMax = MsgRedisKey.STR_IP_ADDR_MAX_SMS.copy().appendKey(param.getIpAddr());
            long maxTime = redisStringAdapter.getLong(ipAddrMax);
            if (++maxTime > messageProperties.getIpAddrMaxTimes()) {
                log.error("当前ip超过每日发送验证码短信限额,发送参数：{}", param);
                return Message.build(false).setMessage("短信发送失败");
            }
        }

        // 设备限制
        if (StringUtils.isNotBlank(param.getDevcId()) && !Objects.equals(NONE_DEIVCE_ID, param.getDevcId())) {
            EquipmentDTO equipmentDTO = equipmentFacadeService.getByDeviceId(param.getDevcId());
            if (null == equipmentDTO) {
                log.error("设备不存在，禁止发送短信,发送参数：{}", param);
                return Message.build(false).setMessage("短信发送失败");
            }

            KeyGenerator deviceMax = MsgRedisKey.STR_DEVICE_ADDR_MAX_SMS.copy().appendKey(param.getDevcId());
            long maxTime = redisStringAdapter.getLong(deviceMax);

            if (++maxTime > messageProperties.getDeviceMaxTimes()) {
                log.error("当前设备超过每日发送短信限额,发送参数：{}", param);
                return Message.build(false).setMessage("短信发送失败");
            }
        }

        // 手机号限额
        KeyGenerator mobileMax = STR_MOBILE_MAX_SMS.copy().appendKey(param.getPhone());
        long maxTime = redisStringAdapter.getLong(mobileMax);
        if (++maxTime > messageProperties.getPhoneMaxTimes()) {
            log.error("手机号超过每日发送短信限额,发送参数：{}", param);
            return Message.build(false).setMessage("短信发送失败");
        }

        if (redisStringAdapter.hasKey(key)) {
            log.error("发送过于频繁，请稍后再试,发送参数：{}", param);
            return Message.build(false).setMessage("短信发送失败");
        }

        return Message.build();
    }

    /**
     * 记录总的请求次数
     * ip 设备 手机号
     *
     * @param param
     */
    private void incrementTempTotalTimes(SendSmsCodeParam param) {
        String str = DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(new Date());

        redisHashMapAdapter.increment(MsgRedisKey.STR_IP_ADDR_MAX_SMS.copy().appendKey("totalCount").appendKey(str),
                param.getIpAddr(), 1);
        redisHashMapAdapter.increment(MsgRedisKey.STR_DEVICE_ADDR_MAX_SMS.copy().appendKey("totalCount").appendKey(str),
                param.getDevcId(), 1);
        redisHashMapAdapter.increment(MsgRedisKey.STR_MOBILE_MAX_SMS.copy().appendKey("totalCount").appendKey(str),
                param.getPhone(), 1);
    }

    /**
     * 发送成功后的次数
     * ip 设备 手机号
     *
     * @param param
     */
    private void incrementTempSuccessTimes(SendSmsCodeParam param) {
        String str = DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(new Date());

        redisHashMapAdapter.increment(MsgRedisKey.STR_IP_ADDR_MAX_SMS.copy().appendKey("successCount").appendKey(str),
                param.getIpAddr(), 1);
        redisHashMapAdapter.increment(MsgRedisKey.STR_DEVICE_ADDR_MAX_SMS.copy().appendKey("successCount").appendKey(str),
                param.getDevcId(), 1);
        redisHashMapAdapter.increment(MsgRedisKey.STR_MOBILE_MAX_SMS.copy().appendKey("successCount").appendKey(str),
                param.getPhone(), 1);
    }


    private void incrementTimes(SendSmsCodeParam param) {

        // ip + 1
        KeyGenerator ipAddrMax = MsgRedisKey.STR_IP_ADDR_MAX_SMS.copy().appendKey(param.getIpAddr());
        redisStringAdapter.increment(ipAddrMax, 1);
        redisStringAdapter.expire(ipAddrMax, DateUtils.getCurSeconds());

        // 设备 + 1
        KeyGenerator deviceMax = MsgRedisKey.STR_DEVICE_ADDR_MAX_SMS.copy().appendKey(param.getDevcId());
        redisStringAdapter.increment(deviceMax, 1);
        redisStringAdapter.expire(deviceMax, DateUtils.getCurSeconds());

        // 手机号 + 1
        KeyGenerator mobileMax = STR_MOBILE_MAX_SMS.copy().appendKey(param.getPhone());
        redisStringAdapter.increment(mobileMax, 1);
        redisStringAdapter.expire(mobileMax, DateUtils.getCurSeconds());
    }

    /**
     * 判断是否是特殊指定的手机号
     *
     * @param mobile
     * @return
     */
    private boolean isSpecificPhone(String mobile) {

        String res = messageProperties.getSpecificMobiles();

        if (StringUtils.isNotBlank(res)) {
            List<String> mobiles = Arrays.asList(res.split(","));

            return mobiles.stream()
                    .map(p -> {
                        p.replace(" ", "");
                        return p;
                    })
                    .filter(p -> StringUtils.isNotBlank(p) && p.equals(mobile))
                    .count() > 0;
        }
        return false;
    }


    /**
     * 发送验证码短信
     *
     * @param phone   手机号
     * @param content 带验证码的短信
     */
    abstract void doSendSmsCode(String phone, String content);

    /**
     * 带变量的发送
     *
     * @param phone   手机号
     * @param content 带占位符的的短信模板
     * @param args    参数 可为null
     */
    abstract void doSendVariableSms(String phone, String content, Object... args);

    /**
     * 保存发送记录
     *
     * @param entity
     */
    protected void saveSendRecord(SmsBean entity) {
        entity.setId(sequenceCreater.nextLongId());
        entity.setCreateTime(new Date());
        entity.setSendTime(new Date());
        entity.setPlatform(this.name());
        msgSmsMapper.insert(entity);
    }

    @Override
    public Message multiSend(List<String> phoneList, String content) {
        return null;
    }
}
