package com.bxm.foundation.config.advert.service.thirdparty.facade;

import com.alibaba.fastjson.JSON;
import com.bxm.foundation.config.advert.entity.thirdparty.ThirdpartyAdvertClickHistoryEntity;
import com.bxm.foundation.config.advert.facade.service.ThirdpartyAdvertCallbackService;
import com.bxm.foundation.config.advert.service.config.ThirdpartyProperties;
import com.bxm.foundation.config.advert.service.enums.CallbackEventType;
import com.bxm.foundation.config.advert.service.thirdparty.callback.AdvertCallback;
import com.bxm.foundation.config.advert.thirdparty.mapper.ThirdpartyAdvertClickHistoryMapper;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.DubboService;

import java.util.Calendar;
import java.util.Date;
import java.util.Map;

import static com.bxm.foundation.config.advert.service.constant.ThirdpartyRedisConfig.*;
import static com.bxm.newidea.component.tools.DateUtils.addField;

/**
 * @author lowi
 * @date 2021/6/15 20:22
 */
@Slf4j
@DubboService(version = "1.0.0", protocol = {"dubbo"})
@AllArgsConstructor
public class ThirdpartyAdvertCallbackServiceImpl implements ThirdpartyAdvertCallbackService {

    private final RedisSetAdapter redisSetAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final ThirdpartyProperties thirdpartyProperties;

    private final ThirdpartyAdvertClickHistoryMapper advertClickHistoryMapper;

    private final AdvertCallback advertCallback;


    @Override
    public void active(String equipmentId, Long userId, String srcApp) {
        ThirdpartyAdvertClickHistoryEntity matchEquipment = getMatchEquipment(equipmentId, CallbackEventType.ACTIVE);

        if (matchEquipment == null) {
            return;
        }
        matchEquipment.setEffectTime(new Date());
        advertCallback.active(matchEquipment);

        if (equipmentId == null) {
            return;
        }

        Long equipment = Long.valueOf(equipmentId);

        KeyGenerator keyGenerator = ADVERT_ACTIVE_CACHE.copy().appendKey(equipment % 100);

        Date firstActiveDate = redisHashMapAdapter.get(keyGenerator, equipmentId, Date.class);

        // 首次激活，则记录数据，进行次留上报
        if (null == firstActiveDate) {
            redisHashMapAdapter.put(keyGenerator, equipment.toString(), new Date());
        }

        log.debug("设备 {} 首次活跃日期：{}", equipment, DateUtils.formatDateTime(firstActiveDate));

        if (null != firstActiveDate && DateUtils.isSameDay(addField(firstActiveDate, Calendar.DAY_OF_YEAR, 1), new Date())) {
            log.info("符合条件，上报次留数据：{}", equipmentId);

            KeyGenerator key = ADVERT_NEXT_DAY_LEFT_CACHE.copy().appendKey(DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));

            if (redisSetAdapter.exists(key, equipmentId)) {
                log.info("{} 今日已上报次日留存，不做处理", equipment);
            } else {
                ThirdpartyAdvertClickHistoryEntity matchRes = advertClickHistoryMapper.selectByEquipment(equipment);

                log.info("处理次日留存：{}", JSON.toJSONString(matchRes));

                if (matchRes != null) {
                    advertCallback.nextDayLeft(matchRes);
                    redisSetAdapter.add(key, equipmentId);
                    redisSetAdapter.expire(key, DateUtils.getCurSeconds());
                }
            }
        }

    }


    /**
     * 判断触发事件的设备是否为广告设备
     *
     * @param equipmentId 设备ID
     * @return 匹配结果，如果为null表示不是广告点击的数据
     */
    private ThirdpartyAdvertClickHistoryEntity getMatchEquipment(String equipmentId, CallbackEventType eventType) {
        if (thirdpartyProperties.isEnableAdvert()) {
            if (!redisSetAdapter.exists(ADVERT_CLICK_CACHE, equipmentId)) {
                log.debug("{} 不是广告点击设备，不做处理", equipmentId);
                return null;
            }
            Map<String, Boolean> entries = getEventMap(equipmentId);
            if (entries.get(eventType.name()) == null) {
                log.debug("新增了上报{}事件,增加事件缓存", eventType.name());
                entries = addEventMap(equipmentId, eventType);
            }
            if (entries.get(eventType.name())) {
                log.debug("设备[{}]已经上报过{}事件", equipmentId, eventType.name());
                return null;
            }

            ThirdpartyAdvertClickHistoryEntity matchRes = advertClickHistoryMapper.selectByEquipment(Long.valueOf(equipmentId));
            log.debug("匹配设备信息，回调广告平台，设备ID：{}", equipmentId);

            if (null != matchRes) {

                matchRes.setMatchMap(entries);
                return matchRes;
            }
        }
        return null;
    }

    /**
     * 为后续增加了枚举类做操作
     *
     * @param equipmentId       设备id
     * @param callbackEventType 类型
     * @return map
     */
    private Map<String, Boolean> addEventMap(String equipmentId, CallbackEventType callbackEventType) {
        KeyGenerator key = ADVERT_EVENT_CALLBACK_CACHE.copy().appendKey(equipmentId);
        redisHashMapAdapter.put(key, callbackEventType.name(), false);
        Map<String, Boolean> entries = Maps.newHashMap();
        entries.put(callbackEventType.name(), false);
        return entries;
    }

    private Map<String, Boolean> getEventMap(String equipmentId) {
        KeyGenerator key = ADVERT_EVENT_CALLBACK_CACHE.copy().appendKey(equipmentId);

        Map<String, Boolean> entries = redisHashMapAdapter.entries(key, Boolean.class);

        if (null == entries || entries.size() == 0) {
            entries = Maps.newHashMap();

            entries.put(CallbackEventType.ACTIVE.name(), false);
            entries.put(CallbackEventType.LOGIN.name(), false);
            entries.put(CallbackEventType.PAYMENT.name(), false);
            entries.put(CallbackEventType.CRUX_BEHAVIORS.name(), false);

            redisHashMapAdapter.putAll(key, entries);
        }

        return entries;
    }
}
