package com.bxm.thirdparty.payment.config;

import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.thirdparty.payment.bo.PreCheckResultBO;
import com.bxm.thirdparty.payment.callback.PaymentCallbackService;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * 引入当前模块后，暴露一些接口，供支付服务进行回调
 * 为了防止发生相互依赖，这里使用rest api暴露接口
 *
 * @author liujia
 * @date 9/12/21 4:29 PM
 **/
@RestController
@RequestMapping("expose/payment")
@Slf4j
public class PaymentExposeController implements ApplicationRunner {

    @GetMapping("checkRepeat")
    public PreCheckResultBO checkRepeat(String orderType, String bizId) {
        return loadInstance(orderType).checkRepeat(bizId);
    }

    @GetMapping("paymentSuccess")
    public void paymentSuccess(String orderType, String bizId) {
        loadInstance(orderType).paymentSuccess(bizId);
    }

    @GetMapping("paymentFail")
    public void paymentFail(String orderType, String bizId, String errorMessage) {
        loadInstance(orderType).paymentFail(bizId, errorMessage);
    }

    private static Map<String, PaymentCallbackService> instanceMap = Maps.newHashMap();

    /**
     * 获取订单类型对应的支付生命周期服务接口
     *
     * @param orderType 订单类型
     * @return 订单类型对应的生命周期处理
     */
    private PaymentCallbackService loadInstance(String orderType) {
        PaymentCallbackService paymentCallbackService = instanceMap.get(orderType);
        if (null == paymentCallbackService) {
            log.error("请在配置文件中配置【payment.config.paymentTypeMap.{}】对应的服务", orderType);

            throw new IllegalArgumentException("未配置[" + orderType + "]对应的PaymentPhaseService");
        }

        return paymentCallbackService;
    }

    @Override
    public void run(ApplicationArguments args) {
        Collection<PaymentCallbackService> phaseServices = SpringContextHolder.getBeans(PaymentCallbackService.class);

        Set<String> orderTypeSet = Sets.newHashSet();

        for (PaymentCallbackService phaseService : phaseServices) {
            log.info("注册订单类型[{}]对应的逻辑处理", phaseService.support());

            if (!orderTypeSet.add(phaseService.support())) {
                throw new IllegalArgumentException("[{}]定义重复，一个订单类型只能定义一个回调类");
            }

            instanceMap.put(phaseService.support(), phaseService);
        }
    }
}
