package com.bxm.newidea.component.apollo.i18n;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.spring.config.ConfigPropertySource;
import com.ctrip.framework.apollo.spring.config.ConfigPropertySourceFactory;
import com.ctrip.framework.apollo.spring.util.SpringInjector;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.support.AbstractMessageSource;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

/**
 * 基于apollo实现可配置刷新的国际化消息
 * @author liujia
 * @date 2019/11/26 17:40
 */
@Component
@Slf4j
public class ApolloMessageSource extends AbstractMessageSource implements InitializingBean {

    /**
     * 默认的国际化区域
     */
    private Locale defaultLocale = Locale.CHINA;

    private final Object lock = new Object();

    private final ConfigPropertySourceFactory configPropertySourceFactory
            = SpringInjector.getInstance(ConfigPropertySourceFactory.class);

    private static final String I18N_NAMESPACE_PREFIX = "i18N";

    private Map<Locale, Map<String, MessageFormat>> cacheMessages = Maps.newHashMap();

    @Override
    protected MessageFormat resolveCode(String code, Locale locale) {
        Map<String, MessageFormat> cacheMap = cacheMessages.get(locale);

        //指定语言的映射关系不存在，尝试加载
        if (null == cacheMap) {
            cacheMap = Maps.newHashMap();
            String loadNameSpace = I18N_NAMESPACE_PREFIX + "." + getLocaleSpace(locale);

            Config config = ConfigService.getConfig(loadNameSpace);

            // 将配置信息转换成map
            if (null != config) {
                ConfigPropertySource configPropertySource = configPropertySourceFactory.getConfigPropertySource(loadNameSpace, config);
                configPropertySource.addChangeListener(changeEvent -> refresh());

                String[] propertyNames = configPropertySource.getPropertyNames();

                for (String propertyName : propertyNames) {
                    String propertyValue = Objects.toString(configPropertySource.getProperty(propertyName));
                    cacheMap.put(propertyName, new MessageFormat(propertyValue, locale));
                }
            }

            cacheMessages.putIfAbsent(locale, cacheMap);
        }

        MessageFormat message = cacheMap.get(code);
        if (null != message) {
            return message;
        }

        //获取默认区域的消息
        cacheMap = cacheMessages.get(defaultLocale);
        if (null != cacheMap) {
            message = cacheMap.get(code);
            if (null != message) {
                log.warn("编码：{}未定义，使用默认的locale", code, locale.toString());
                return message;
            }
        }

        log.warn("编码：{}未定义，默认locale中也未定义", code, locale.toString());
        return null;
    }

    private String getLocaleSpace(Locale locale) {
        return locale.getLanguage() + "_" + locale.getCountry();
    }

    /**
     * 配置变更时重新加载
     */
    private void refresh() {
        synchronized (lock) {
            cacheMessages = Maps.newHashMap();
        }
    }

    @Override
    public void afterPropertiesSet() {
        MessageHelper.setMessageSource(this);
    }
}




















