package com.bxm.localnews.common.vo;

import com.bxm.localnews.common.constant.RespCode;
import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

/**
 * 输出的标准Json格式数据， 用于替代原本黄瓜头条里面的Object输出
 * <code>Json.gen().data("mydata").code(Response.OK).output()</code>
 * @param <T> 泛型，Json中包含的具体的数据格式
 * @author liujia
 */
@ApiModel(description = "标准的Json响应格式数据")
public class Json<T> {

    @ApiModelProperty(required = true, value = "操作结果编码", example = "0/400/401/403/404/500")
    private int code;

    @ApiModelProperty(value = "请求的返回消息，code为0可能为空，其他code均有值", example = "操作完成/操作成功/余额不足...")
    private String errorMsg = StringUtils.EMPTY;

    @ApiModelProperty(value = "请求的返回数据，根据请求不同有具体的返回实体")
    private T result;

    @ApiModelProperty(value = "如果当前请求是带有分页属性，则返回分页相关的参数信息(对旧版本接口兼容，1.1.0版本后的接口不支持)",hidden = true)
    @Deprecated
    private PageModel page;

    protected Json() {
        this.code = RespCode.OK;
    }

    protected Json(int code, String errorMsg) {
        this.code = code;
        this.errorMsg = errorMsg;
    }

    protected Json(Message message) {
        this.code = message.isSuccess() ? RespCode.OK : RespCode.BAD_REQUEST;
        this.errorMsg = message.getLashMessaage();
    }

    @Deprecated
    protected Json(int code, String errorMsg, T result, PageModel page) {
        this.code = code;
        this.errorMsg = errorMsg;
        this.result = result;
        this.page = page;
    }

    protected Json(int code, String errorMsg, T result) {
        this.code = code;
        this.errorMsg = errorMsg;
        this.result = result;
    }

    protected Json(Json<T> source) {
        this.code = source.code;
        this.errorMsg = source.errorMsg;
        this.result = source.result;
        this.page = source.page;
    }

    public static <T> Json<T> build() {
        return new Json<>();
    }

    public static <T> Json<T> build(T result) {
        Json<T> json = new Json<>();
        return json.setResult(result);
    }

    public static <T> Json<T> build(Message message) {
        return new Json<>(message);
    }

    public static <T> Json<T> build(int code, String errorMsg) {
        return new Json<>(code, errorMsg);
    }

    public static <T> Json<T> build(int code, String errorMsg, T result) {
        return new Json<>(code, errorMsg, result);
    }

    @Deprecated
    public static <T> Json<T> build(int code, String errorMsg, T result, PageModel page) {
        return new Json<>(code, errorMsg, result, page);
    }

    /**
     * 操作是否成功
     * @return true表示操作成功
     */
    public boolean success() {
        return this.code == RespCode.OK;
    }

    public int getCode() {
        return this.code;
    }

    public Json<T> setCode(int code) {
        this.code = code;
        return this;
    }

    public String getErrorMsg() {
        return this.errorMsg;
    }

    public Json<T> setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public T getResult() {
        return this.result;
    }

    public Json<T> setResult(T result) {
        this.result = result;
        return this;
    }

    @Deprecated
    public PageModel getPage() {
        return this.page;
    }

    @Deprecated
    public Json<T> setPage(PageModel page) {
        this.page = page;
        return this;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
    }

}
