package com.bxm.newidea.component.tools;

import java.util.Arrays;
import java.util.List;

/**
 * 二进制操作工具类
 * @author liujia
 */
public class BitOperatorUtil {

    /**
     * 把一个整形转化为byte
     */
    public static byte integerTo1Byte(int value) {
        return (byte) (value & 0xFF);
    }

    /**
     * 把一个整形转化为1位的byte数组
     */
    public static byte[] integerTo1Bytes(int value) {
        byte[] result = new byte[1];
        result[0] = (byte) (value & 0xFF);
        return result;
    }

    /**
     * 把一个整形改为2位的byte数组
     */
    public static byte[] integerTo2Bytes(int value) {
        byte[] result = new byte[2];
        result[0] = (byte) ((value >>> 8) & 0xFF);
        result[1] = (byte) (value & 0xFF);
        return result;
    }

    /**
     * 把一个整形改为3位的byte数组
     */
    public static byte[] integerTo3Bytes(int value) {
        byte[] result = new byte[3];
        result[0] = (byte) ((value >>> 16) & 0xFF);
        result[1] = (byte) ((value >>> 8) & 0xFF);
        result[2] = (byte) (value & 0xFF);
        return result;
    }

    /**
     * 把一个整形改为4位的byte数组
     */
    public static byte[] integerTo4Bytes(int value) {
        byte[] result = new byte[4];
        result[0] = (byte) ((value >>> 24) & 0xFF);
        result[1] = (byte) ((value >>> 16) & 0xFF);
        result[2] = (byte) ((value >>> 8) & 0xFF);
        result[3] = (byte) (value & 0xFF);
        return result;
    }

    /**
     * 把byte[]转化位整形,通常为指令用
     */
    public static int byteToInteger(byte[] value) {
        int result;
        if (value.length == 1) {
            result = oneByteToInteger(value[0]);
        } else if (value.length == 2) {
            result = twoBytesToInteger(value);
        } else if (value.length == 3) {
            result = threeBytesToInteger(value);
        } else if (value.length == 4) {
            result = fourBytesToInteger(value);
        } else {
            result = fourBytesToInteger(value);
        }
        return result;
    }

    public static byte[] short2Byte(short number) {
        int temp = number;
        byte[] b = new byte[2];
        for (int i = 0; i < b.length; i++) {
            //将最低位保存在最低位
            b[i] = new Integer(temp & 0xff).byteValue();
            // 向右移8位
            temp = temp >> 8;
        }
        return b;
    }

    /**
     * 把一个byte转化位整形,通常为指令用
     */
    public static int oneByteToInteger(byte value) {
        return (int) value & 0xFF;
    }

    /**
     * 把一个2位的数组转化位整形
     */
    public static int twoBytesToInteger(byte[] value) {
        int temp0 = value[0] & 0xFF;
        int temp1 = value[1] & 0xFF;
        return ((temp0 << 8) + temp1);
    }

    /**
     * 把一个3位的数组转化位整形
     */
    public static int threeBytesToInteger(byte[] value) {
        int temp0 = value[0] & 0xFF;
        int temp1 = value[1] & 0xFF;
        int temp2 = value[2] & 0xFF;
        return ((temp0 << 16) + (temp1 << 8) + temp2);
    }

    /**
     * 把一个4位的数组转化位整形,通常为指令用
     */
    public static int fourBytesToInteger(byte[] value) {
        int temp0 = value[0] & 0xFF;
        int temp1 = value[1] & 0xFF;
        int temp2 = value[2] & 0xFF;
        int temp3 = value[3] & 0xFF;
        return ((temp0 << 24) + (temp1 << 16) + (temp2 << 8) + temp3);
    }

    /**
     * 把一个4位的数组转化位整形
     */
    public static long fourBytesToLong(byte[] value) throws Exception {
        // if (value.length < 4) {
        // throw new Exception("Byte array too short!");
        // }
        int temp0 = value[0] & 0xFF;
        int temp1 = value[1] & 0xFF;
        int temp2 = value[2] & 0xFF;
        int temp3 = value[3] & 0xFF;
        return (((long) temp0 << 24) + (temp1 << 16) + (temp2 << 8) + temp3);
    }

    /**
     * 把一个数组转化长整形
     */
    public static long bytes2Long(byte[] value) {
        long result = 0;
        int len = value.length;
        int temp;
        for (int i = 0; i < len; i++) {
            temp = (len - 1 - i) * 8;
            if (temp == 0) {
                result += (value[i] & 0x0ff);
            } else {
                result += (value[i] & 0x0ff) << temp;
            }
        }
        return result;
    }

    /**
     * 把一个长整形改为byte数组
     */
    public static byte[] longToBytes(long value) {
        return longToBytes(value, 8);
    }

    /**
     * 把一个长整形改为byte数组
     */
    public static byte[] longToBytes(long value, int len) {
        byte[] result = new byte[len];
        int temp;
        for (int i = 0; i < len; i++) {
            temp = (len - 1 - i) * 8;
            if (temp == 0) {
                result[i] += (value & 0x0ff);
            } else {
                result[i] += (value >>> temp) & 0x0ff;
            }
        }
        return result;
    }

    /**
     * 合并字节数组
     */
    public static byte[] concatAll(byte[] first, byte[]... rest) {
        int totalLength = first.length;
        for (byte[] array : rest) {
            if (array != null) {
                totalLength += array.length;
            }
        }
        byte[] result = Arrays.copyOf(first, totalLength);
        int offset = first.length;
        for (byte[] array : rest) {
            if (array != null) {
                System.arraycopy(array, 0, result, offset, array.length);
                offset += array.length;
            }
        }
        return result;
    }

    /**
     * 合并字节数组
     */
    public static byte[] concatAll(List<byte[]> rest) {
        int totalLength = 0;
        for (byte[] array : rest) {
            if (array != null) {
                totalLength += array.length;
            }
        }
        byte[] result = new byte[totalLength];
        int offset = 0;
        for (byte[] array : rest) {
            if (array != null) {
                System.arraycopy(array, 0, result, offset, array.length);
                offset += array.length;
            }
        }
        return result;
    }

    public static float byte2Float(byte[] bs) {
        return Float.intBitsToFloat(
                (((bs[3] & 0xFF) << 24) + ((bs[2] & 0xFF) << 16) + ((bs[1] & 0xFF) << 8) + (bs[0] & 0xFF)));
    }

    public static float byteBE2Float(byte[] bytes) {
        int l;
        l = bytes[0];
        l &= 0xff;
        l |= ((long) bytes[1] << 8);
        l &= 0xffff;
        l |= ((long) bytes[2] << 16);
        l &= 0xffffff;
        l |= ((long) bytes[3] << 24);
        return Float.intBitsToFloat(l);
    }

    public static int getCheckSum4JT808(byte[] bs, int start, int end) {
        if (start < 0 || end > bs.length) {
            throw new ArrayIndexOutOfBoundsException("getCheckSum4JT808 error : index out of bounds(start=" + start
                    + ",end=" + end + ",bytes length=" + bs.length + ")");
        }
        int cs = 0;
        for (int i = start; i < end; i++) {
            cs ^= bs[i];
        }
        return cs;
    }

    public static int getBitRange(int number, int start, int end) {
        if (start < 0) {
            throw new IndexOutOfBoundsException("min index is 0,but start = " + start);
        }
        if (end >= Integer.SIZE) {
            throw new IndexOutOfBoundsException("max index is " + (Integer.SIZE - 1) + ",but end = " + end);
        }

        return (number << Integer.SIZE - (end + 1)) >>> Integer.SIZE - (end - start + 1);
    }

    public static int getBitAt(int number, int index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("min index is 0,but " + index);
        }
        if (index >= Integer.SIZE) {
            throw new IndexOutOfBoundsException("max index is " + (Integer.SIZE - 1) + ",but " + index);
        }

        return ((1 << index) & number) >> index;
    }

    public static int getBitAtS(int number, int index) {
        String s = Integer.toBinaryString(number);
        return Integer.parseInt(s.charAt(index) + "");
    }

    /**
     * 判断long类型中的某一个索引位置是否为1
     * @param number 待判断的值
     * @param index  索引位置，从0开始
     * @return 如果为1则返回true
     */
    public static boolean getBitAsBoolean(Long number, int index) {
        if (null == number) {
            return false;
        }
        return (number & (1L << index)) != 0;
    }

    /**
     * 将long类型中的二进制中的第{index}位设置为1
     * @param number 待设置的数
     * @param index  待设定的索引，从0开始
     * @return 设置完成后的数
     */
    public static long setBitToLong(Long number, int index) {
        if (null == number) {
            number = 0L;
        }
        return number | (1L << index);
    }

    /**
     * 清理long类型中二进制中的第{index}位，将其设置为0
     * @param number 待设置的数
     * @param index  待设定的索引，从0开始
     * @return 清理完成后的值
     */
    public static long clearBit(Long number, int index) {
        if (null == number) {
            number = 0L;
        }
        //000100 按位取反111011
        long mask = ~(1L << index);
        //111011
        return (number & (mask));
    }

    @Deprecated
    public static int getBitRangeS(int number, int start, int end) {
        String s = Integer.toBinaryString(number);
        StringBuilder sb = new StringBuilder(s);
        while (sb.length() < Integer.SIZE) {
            sb.insert(0, "0");
        }
        String tmp = sb.reverse().substring(start, end + 1);
        sb = new StringBuilder(tmp);
        return Integer.parseInt(sb.reverse().toString(), 2);
    }

    /**
     * 判断原始数据中，指定的bit位是否为1
     * 原始数据中每一个Long类型表示63个bit为（从0开始）
     * @param sourceArray 原始数据数组
     * @param bitIndex    二进制位，从0开始，不支持负数
     * @return true表示指定位置bit等于1
     */
    public static boolean hasBit(Long[] sourceArray, int bitIndex) {
        if (null == sourceArray || bitIndex < 0) {
            return false;
        }

        int index = (bitIndex + 1) % 63 == 0 ? (bitIndex + 1) / 63 - 1 : (bitIndex + 1) / 63;

        if (index < sourceArray.length) {
            //获取指定索引位置的授权码
            long partCode = sourceArray[index];
            //获取偏移量，用于对权限码进行位移操作
            int offset;
            if ((bitIndex - index) / 62 > index) {
                offset = 62;
            } else if (bitIndex > 62 && (bitIndex - index) / 62 == index) {
                offset = 0;
            } else {
                offset = ((bitIndex - index) % 62);
            }
            // 将1左移指定的偏移量，将偏移结果和鉴权码进行&操作，如果不等于0，表示该权限位为1
            return BitOperatorUtil.getBitAsBoolean(partCode, offset);
        }
        return false;
    }

    /**
     * 设置原始数据数组中指定bit位
     * 原始数据中每一个Long类型表示63个bit为（从0开始）
     * @param sourceArray 原始数据数组
     * @param bitIndex    二进制位，从0开始，不支持负数
     * @param addOrRemove true表示设置为1，false表示设置为0
     * @return 设置更新后的数组
     */
    public static Long[] setBitToArray(Long[] sourceArray, int bitIndex, boolean addOrRemove) {
        if (null == sourceArray || bitIndex < 0) {
            return sourceArray;
        }

        int length = sourceArray.length;

        int index = (bitIndex + 1) % 63 == 0 ? (bitIndex + 1) / 63 - 1 : (bitIndex + 1) / 63;

        int maxLength = index + 1 > length ? index + 1 : length;
        Long[] afterSet = new Long[maxLength];

        for (int i = 0; i <= index; i++) {
            Long partVal;
            if (i == index) {
                int offset;
                if ((bitIndex - index) / 62 > index) {
                    offset = 62;
                } else if (bitIndex > 62 && (bitIndex - index) / 62 == index) {
                    offset = 0;
                } else {
                    offset = (bitIndex - index) % 62;
                }

                long userAuthCode;
                //超出现有数组下标
                if (i > length - 1) {
                    userAuthCode = 0L;
                } else {
                    userAuthCode = sourceArray[i];
                }

                if (addOrRemove) {
                    partVal = BitOperatorUtil.setBitToLong(userAuthCode, offset);
                } else {
                    partVal = BitOperatorUtil.clearBit(userAuthCode, offset);
                }
            } else if (i > length - 1) {
                partVal = 0L;
            } else {
                partVal = sourceArray[i];
            }

            afterSet[i] = partVal;
        }

        return afterSet;
    }
}



