package com.bxm.newidea.component.tools;

import lombok.experimental.UtilityClass;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

import static com.bxm.newidea.component.tools.DateUtils.*;

/**
 * 日期美化工具类
 *
 * @author liujia
 * @date 6/7/21 3:32 PM
 **/
@UtilityClass
public class DateBeautifyUtils {

    private static final String ONE_SECOND_AGO = "几秒前";

    private static final String ONE_MINUTE_AGO = "分钟前";

    private static final String ONE_HOUR_AGO = "小时前";

    private static final String YESTERDAY = "昨天";

    private static final String ONE_DAY_AGO = "天前";

    private static final String ONE_WEEK_AGO = "周前";

    private static final String ONE_MONTH_AGO = "月前";

    private static final String ONE_YEAR_AGO = "年前";

    /**
     * 时间转换成指定的格式
     * 根据当前时间与传入的时间对比，返回一个时间描述字符串
     *
     * @param date 需要转换的时间
     * @return 时间描述字符串，如：刚刚、几分钟前、几小时前、昨天何时、前天何时等
     */
    public static String timeConvertString(Date date) {
        if (Objects.isNull(date)) {
            date = new Date();
        }
        //转化的时间戳
        long coverTimeMillis = date.getTime();
        //当前时间戳
        Calendar instance = Calendar.getInstance();
        //昨天零点时间
        Calendar yesterday = getMorningTime(-1);
        long yesterdayTimeInMillis = yesterday.getTimeInMillis();
        //前天零点时间
        Calendar beforeYesterday = getMorningTime(-2);
        //当前时间戳
        long currentTimeMillis = instance.getTimeInMillis();

        long differenceTime = currentTimeMillis - coverTimeMillis;
        String result;
        //1分钟内
        if (differenceTime <= MINUTE_MILLISECOND) {
            result = "刚刚";
            //1小时内
        } else if (differenceTime > MINUTE_MILLISECOND && differenceTime <= HOUR_MILLISECOND) {
            result = differenceTime / MINUTE_MILLISECOND + "分钟前";
            //一天内
        } else if (differenceTime > HOUR_MILLISECOND && differenceTime < DAY_MILLISECOND) {
            result = differenceTime / HOUR_MILLISECOND + "小时前";
            //昨天
        } else if (differenceTime >= DAY_MILLISECOND && differenceTime <= currentTimeMillis - yesterdayTimeInMillis) {
            result = "昨天" + DateFormatUtils.format(date, "HH:mm");
        } else if (differenceTime > currentTimeMillis - yesterdayTimeInMillis && differenceTime <= currentTimeMillis - beforeYesterday.getTimeInMillis()) {
            result = "前天" + DateFormatUtils.format(date, "HH:mm");
        } else {
            result = DateFormatUtils.format(date, "yyyy/MM/dd");
        }
        return result;
    }

    /**
     * 获取传入时间参数与当前时间差,使用一种比较自然的语言描述（eg:一分钟之前）
     *
     * @param date 要比较的时间
     * @return 返回结果字符串
     */
    public static String getTimeLag(Date date) {
        if (null == date) {
            return null;
        }
        //获取时间差
        long delta = System.currentTimeMillis() - date.getTime();

        if (delta < MINUTE_MILLISECOND) {
            return ONE_SECOND_AGO;
        }
        if (delta < HOUR_MILLISECOND) {
            long minutes = delta / MINUTE_MILLISECOND;
            return minutes + ONE_MINUTE_AGO;
        }
        if (delta < DAY_MILLISECOND) {
            long hours = delta / HOUR_MILLISECOND;
            return hours + ONE_HOUR_AGO;
        }
        if (delta < 2L * DAY_MILLISECOND) {
            return YESTERDAY;
        }
        if (delta < WEEK_MILLISECOND) {
            long days = delta / DAY_MILLISECOND;
            return days + ONE_DAY_AGO;
        }
        if (delta < 30L * DAY_MILLISECOND) {
            long days = delta / WEEK_MILLISECOND;
            return days + ONE_WEEK_AGO;
        }
        if (delta < 12L * 4L * WEEK_MILLISECOND) {
            long months = delta / (30L * DAY_MILLISECOND);
            return months + ONE_MONTH_AGO;
        } else {
            long years = delta / (365L * DAY_MILLISECOND);
            return (years <= 0 ? 1 : years) + ONE_YEAR_AGO;
        }
    }

    /**
     * 获得零点时间
     */
    private static Calendar getMorningTime(Integer amount) {
        Calendar morningTime = Calendar.getInstance();
        morningTime.set(Calendar.HOUR_OF_DAY, 0);
        morningTime.set(Calendar.MINUTE, 0);
        morningTime.set(Calendar.SECOND, 0);
        morningTime.set(Calendar.MILLISECOND, 0);
        morningTime.add(Calendar.DAY_OF_YEAR, amount);
        return morningTime;
    }

    /**
     * 根据传入的日期，使用一种比较自然的语言描述 （eg：前天，昨天，今天，明天，后天，某某日期）
     *
     * @return
     */
    public static String getDayLag(Date date) {

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        Calendar now = Calendar.getInstance();

        if (now.get(Calendar.DAY_OF_MONTH) == calendar.get(Calendar.DAY_OF_MONTH)) {
            return "今天";
        }

        if (calendar.after(now)) {
            now.add(Calendar.DAY_OF_MONTH, 1);

            if (now.get(Calendar.DAY_OF_MONTH) == calendar.get(Calendar.DAY_OF_MONTH)) {
                return "明天";
            }

            now.add(Calendar.DAY_OF_MONTH, 1);
            if (now.get(Calendar.DAY_OF_MONTH) == calendar.get(Calendar.DAY_OF_MONTH)) {
                return "后天";
            }

        } else {
            // before
            now.add(Calendar.DAY_OF_MONTH, -1);
            if (now.get(Calendar.DAY_OF_MONTH) == calendar.get(Calendar.DAY_OF_MONTH)) {
                return "昨天";
            }

            now.add(Calendar.DAY_OF_MONTH, -1);
            if (now.get(Calendar.DAY_OF_MONTH) == calendar.get(Calendar.DAY_OF_MONTH)) {
                return "前天";
            }

        }

        // 返回日期
        return formatDate(date);
    }
}
