package com.bxm.newidea.component;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * @author lowi
 * @date 2021/3/3 9:46
 */
@Slf4j
public class JSON {

    private final static ObjectMapper objectMapper;


    static {
        objectMapper = new ObjectMapper();

        //对象属性为空时，默认序列化会失败,FAIL_ON_EMPTY_BEANS设置当对象没有属性时，让其序列化能成功，不抛异常。
        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        //Jackson对时间(Date)序列化的转换格式默认是时间戳，可以取消时间的默认时间戳转化格式；
        // 默认时间戳转化格式取消后在序列化时日期格式默认为 yyyy-MM-dd'T'HH:mm:ss.SSSZ，同时需要设置要展现的时间格式。
        //TODO 加上注解，则会导致序列化抛去毫秒,对于精细的操作可能会有影响，故不设置此配置
//        objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
//        objectMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.sss"));

        //默认情况下Jackson要求JSON字符串消息 和 Java类中的字段必须一一相对应，
        // 否则反序列解析JSON字符串时会报错。当然也可以通过配置Jackson的ObjectMapper属性使Jackson在反序列化时，
        // 忽略在 json字符串中存在但 Java 对象不存在的属性。
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        //java对象序列化后的json串里不出现属性为null的字段
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        //反序列Json字符串中属性名没有双引号,解析器默认不能解析，
        // 需要设置Feature.ALLOW_UNQUOTED_FIELD_NAMES属性来处理这种没有双引号的json串。
        objectMapper.enable(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES);
        //默认情况下objectMapper解析器是不能解析以"0"为开头的数字，需要开启Feature.ALLOW_NUMERIC_LEADING_ZEROS才能使用。
        objectMapper.enable(JsonParser.Feature.ALLOW_NUMERIC_LEADING_ZEROS);
        //浮点类型 或 数字类型都可以接受NaN值，反序列化需要开启Feature.ALLOW_NON_NUMERIC_NUMBERS
        objectMapper.enable(JsonParser.Feature.ALLOW_NON_NUMERIC_NUMBERS);
        //Json反序列化可以解析单引号包住的属性名称和字符串值
        objectMapper.enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);
        //Feature.ALLOW_UNQUOTED_CONTROL_CHARS该特性决定parser是否允许JSON字符串包含非引号控制字符
        // （值小于32的ASCII字符，包含制表符\t、换行符\n和回车\r）。
        // 如果该属性关闭，则如果遇到这些字符，则会抛出异常。
        objectMapper.enable(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS);
    }

    public static ObjectMapper addConfigure(SerializationFeature f, boolean state) {
        objectMapper.configure(f, state);
        return objectMapper;
    }

    public static ObjectMapper addConfigure(DeserializationFeature f, boolean state) {
        objectMapper.configure(f, state);
        return objectMapper;
    }

    public static ObjectMapper addConfigure(JsonInclude.Include jsonInclude) {
        objectMapper.setSerializationInclusion(jsonInclude);
        return objectMapper;
    }

    public static ObjectMapper getObjectMapper() {
        return objectMapper;
    }


    public static String toJSONString(Object value) {
        if (validationValue(value)) {
            return null;
        }
        try {
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            log.warn("JSON.toJSONString出错，请求参数Object：param:{}", value, e);
        }
        return null;
    }


    public static Object parse(String text) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, Object.class);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数String：param:{}", text, e);
        }
        return null;
    }

    public static Object parse(String text, JsonParser.Feature... feature) {
        if (validationValue(text)) {
            return null;
        }
        try {
            ObjectMapper disable = objectMapper.enable(feature);
            return disable.readValue(text, Object.class);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数String：param:{},JsonParser.Feature:{}", text, feature, e);
        }
        return null;
    }

    public static Object parse(byte[] src) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, Object.class);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数byte[]：param:{}", src, e);
        }
        return null;
    }

    public static Object parse(byte[] src, int offset, int len) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, offset, len, Object.class);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数byte[]：{},offset:{},len:{}", src, offset, len, e);
        }
        return null;
    }

    public static <T> Object parse(byte[] src, TypeReference<T> valueTypeRef) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, valueTypeRef);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数byte[]：{},TypeReference<T>:{}", src, valueTypeRef, e);
        }
        return null;
    }

    public static <T> Object parse(byte[] src, int offset, int len, TypeReference<T> valueTypeRef) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, offset, len, valueTypeRef);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数byte[]：{},offset:{},len:{},TypeReference<T>:{}", src, offset, len, valueTypeRef, e);
        }
        return null;
    }

    public static <T> Object parse(byte[] src, JavaType javaType) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, javaType);
        } catch (IOException e) {
            log.warn("JSON.parse出错，请求参数byte[]：{},javaType:{}", src, javaType, e);
        }
        return null;
    }

    public static JSONObject parseObject(String text) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, JSONObject.class);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，请求参数text:{}", text, e);
        }
        return null;
    }

    public static JSONObject parseObject(String text, JsonParser.Feature... feature) {
        if (validationValue(text)) {
            return null;
        }
        try {
            ObjectMapper disable = objectMapper.enable(feature);
            return disable.readValue(text, JSONObject.class);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，请求参数text:{},JsonParser.Feature:{}", text, feature, e);
        }
        return null;
    }

    public static <T> T parseObject(String text, Class<T> clazz) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, clazz);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，请求参数text:{},Class<T>:{}", text, clazz, e);
        }
        return null;
    }

    public static <T> T parseObject(InputStream in, Class<T> clazz) {
        if (validationValue(in)) {
            return null;
        }
        try {
            return objectMapper.readValue(in, clazz);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，InputStream:{},Class<T>:{}", in, clazz, e);
        }
        return null;
    }

    public static <T> T parseObject(byte[] src, Class<T> clazz) {
        if (validationValue(src)) {
            return null;
        }
        try {
            return objectMapper.readValue(src, clazz);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，byte[]:{},Class<T>:{}", src, clazz, e);
        }
        return null;
    }

    public static <T> T toJavaObject(JSONObject json, Class<T> clazz) {
        if (validationValue(json)) {
            return null;
        }
        try {
            return objectMapper.readValue(JSON.toJSONString(json), clazz);
        } catch (Exception e) {
            log.warn("JSON.toJavaObject出错，JSONObject:{},Class<T>:{}", json, clazz, e);
        }
        return null;
    }

    public static <T> T parseArray(String text, TypeReference<T> valueTypeRef) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, valueTypeRef);
        } catch (IOException e) {
            log.warn("JSON.parseArray出错，text:{},TypeReference<T>:{}", text, valueTypeRef, e);
        }
        return null;
    }

    public static <T> List<T> parseArray(String value, Class<T> clazz) {
        if (validationValue(value)) {
            return null;
        }
        JSONArray objects = JSON.parseArray(value);
        List<T> list = new ArrayList<T>(objects.size());
        for (Object item : objects) {
            list.add(JSON.parseObject(JSON.toJSONString(item), clazz));
        }
        return list;
    }

    public static <T> T parseObject(String text, TypeReference<T> valueTypeRef) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, valueTypeRef);
        } catch (IOException e) {
            log.warn("JSON.parseObject出错，text:{},TypeReference<T>:{}", text, valueTypeRef, e);
        }
        return null;
    }

    public static JSONArray parseArray(String text) {
        if (validationValue(text)) {
            return null;
        }
        try {
            return objectMapper.readValue(text, new TypeReference<JSONArray>() {
            });
        } catch (IOException e) {
            log.warn("JSON.parseArray出错，text:{}", text, e);
        }
        return null;
    }

    public static byte[] toJSONBytes(Object obj) {
        if (validationValue(obj)) {
            return null;
        }
        try {
            return objectMapper.writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            log.warn("JSON.toJSONBytes出错，obj:{}", obj, e);
        }
        return null;
    }

    public static Object toJSON(Object obj) {
        if (validationValue(obj)) {
            return null;
        }
        try {
            return objectMapper.readValue(JSON.toJSONString(obj), Object.class);
        } catch (IOException e) {
            log.warn("JSON.toJSON出错，obj:{}", obj, e);
        }
        return null;
    }

    private static boolean validationValue(Object obj) {
        if (obj == null) {
            return true;
        }
        if (obj instanceof String) {
            return "".equals(obj.toString());
        }
        return false;
    }


}
