package com.bxm.component.oncejob.storage.mysql;

import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.enums.ActionEnum;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.IPUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.JdbcTemplate;

import java.util.Date;
import java.util.StringJoiner;

/**
 * 通过mysql实现分布式锁
 *
 * @author liujia
 * @date 8/24/21 7:55 PM
 **/
@Slf4j
@AllArgsConstructor
public class MysqlLock {

    private JdbcTemplate jdbcTemplate;

    private ComponentOnceJobConfigurationProperties properties;

    /**
     * 锁定某一个业务操作，防止实例竞争
     *
     * @param action       业务动作
     * @param timeoutMills 超时时间()
     * @return 锁定结果，如果为false，则说明有其他实例正在执行
     */
    public boolean lock(ActionEnum action, int timeoutMills) {
        try {
            saveActionLock(action, timeoutMills);
        } catch (Exception e) {
            // 插入数据异常时，尝试查询处理
            try {
                return tryRelocked(action, timeoutMills);
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                return false;
            }
        }
        return true;
    }

    /**
     * 尝试重新上锁，判断之前的锁是否过期,如果过期，则删除后重新上锁，如果未过期则上锁失败
     *
     * @param action 业务动作
     * @return 重新加锁后的结果
     */
    private boolean tryRelocked(ActionEnum action, int timeoutMills) {
        Date releaseTime = jdbcTemplate.query("select release_time from t_component_job_lock", t -> {
            return t.getDate("release_time");
        });

        // 如果已经过了释放时间，则删除锁，并重新加锁
        if (DateUtils.before(releaseTime)) {
            String deleteSql = String.format("delete from t_component_job_lock where action = '%s' and app = '%s'",
                    action,
                    properties.getAppName());
            jdbcTemplate.execute(deleteSql);

            log.info("锁未及时释放或设置的操作间隔存在问题，action:{},releaseTime:{}",
                    action,
                    DateUtils.formatDateTime(releaseTime));
        } else {
            if (log.isDebugEnabled()) {
                log.debug("{}已被其他实例加锁", action);
            }

            return false;
        }

        try {
            saveActionLock(action, timeoutMills);
        } catch (Exception e) {
            log.error("二次加锁失败，action:{},app:{}", action, properties.getAppName());
            log.error(e.getMessage());
            return false;
        }

        return true;
    }

    private void saveActionLock(ActionEnum action, int timeoutMills) {
        StringJoiner columnJoiner = new StringJoiner(",", "(", ")");
        StringJoiner valueJoiner = new StringJoiner(",", "(", ")");

        String[] columnArray = new String[]{"app", "action", "lock_time", "release_time", "instance"};

        for (String column : columnArray) {
            columnJoiner.add(column);
            valueJoiner.add("?");
        }

        String sql = "insert into t_component_job_lock " + columnJoiner.toString() + " values " + valueJoiner.toString();
        jdbcTemplate.update(sql, ps -> {
            int index = 1;
            ps.setString(index++, properties.getAppName());
            ps.setString(index++, action.name());
            ps.setDate(index++, new java.sql.Date(System.currentTimeMillis()));
            ps.setDate(index++, new java.sql.Date(System.currentTimeMillis() + timeoutMills));
            ps.setString(index, IPUtil.getLocalRealIp());
        });
    }

    public boolean unlock(String action) {
        String deleteSql = String.format("delete from t_component_job_lock where " +
                        "action = '%s' " +
                        "and app = '%s' " +
                        "and instance = '%s'",
                action,
                properties.getAppName(),
                IPUtil.getLocalRealIp());

        int rows = jdbcTemplate.update(deleteSql);

        if (rows > 0) {
            return true;
        } else {
            log.info("解锁失败，action：{},app:{},instance:{}",
                    action,
                    properties.getAppName(),
                    IPUtil.getLocalRealIp());
            return false;
        }
    }
}
