package com.bxm.component.oncejob.storage.redis;

import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.converter.JobConverter;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.component.oncejob.job.OnceJobDefinition;
import com.bxm.component.oncejob.storage.LongTermJobRepository;
import com.bxm.component.oncejob.utils.JobIdUtil;
import com.bxm.newidea.component.entity.TypedTuple;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Set;

import static com.bxm.component.oncejob.constant.OnceJobRedisKey.*;

/**
 * 通过redis存储长期保存的一次性调度任务
 * 不适合有大量任务的情况下使用
 *
 * @author liujia
 * @date 7/30/21 10:28 AM
 **/
@Slf4j
@AllArgsConstructor
public class RedisLongTermJobRepositoryImpl implements LongTermJobRepository {

    private RedisHashMapAdapter redisHashMapAdapter;

    private ComponentOnceJobConfigurationProperties properties;

    private DistributedLock distributedLock;

    private RedisZSetAdapter redisZSetAdapter;

    @Override
    public String saveJob(OnceJobDefinition definition) {

        JobPersistentObject persistentObject = JobConverter.convert(definition);

        long index = persistentObject.getFireDate() / properties.getFetchLongTermJobMills();

        KeyGenerator key = JOB_LONG_TERM_KEY.copy().appendKey(index);

        redisHashMapAdapter.put(key, persistentObject.getJobId(), persistentObject);

        redisZSetAdapter.add(JOB_LONG_TERM_INDEX_KEY, index, index);

        return persistentObject.getJobId();
    }

    @Override
    public boolean removeJob(String jobId) {
        long fireMills = JobIdUtil.getFireMills(jobId);
        long index = fireMills / properties.getFetchLongTermJobMills();

        KeyGenerator key = JOB_LONG_TERM_KEY.copy().appendKey(index);

        return redisHashMapAdapter.remove(key, jobId) > 0;
    }

    @Override
    public List<JobPersistentObject> clearDirty() {
        String lockKey = JOB_LONG_TERM_DIRTY_LOCK_KEY.gen();

        List<JobPersistentObject> result = Lists.newArrayList();

        if (distributedLock.lock(lockKey)) {
            long index = System.currentTimeMillis() / properties.getFetchLongTermJobMills() + 1;

            Set<TypedTuple<Long>> typedTuples = redisZSetAdapter.rangeByScoreWithScores(JOB_LONG_TERM_INDEX_KEY,
                    0,
                    index,
                    true,
                    Long.class);

            log.info("发现[{}]个未处理的时间片段", typedTuples.size());

            for (TypedTuple<Long> typedTuple : typedTuples) {
                result.addAll(clearInterval(typedTuple.getValue()));
            }

            distributedLock.unlock(lockKey);
        } else {
            log.warn("其他实例正在处理脏数据，放弃本次处理");
        }

        return result;
    }

    private List<JobPersistentObject> clearInterval(long index) {
        KeyGenerator key = JOB_LONG_TERM_KEY.copy().appendKey(index);
        List<JobPersistentObject> values = redisHashMapAdapter.values(key, JobPersistentObject.class);

        if (null != values) {
            redisHashMapAdapter.remove(key);
            redisZSetAdapter.remove(JOB_LONG_TERM_INDEX_KEY, index);

            log.warn("补偿处理历史数据,索引为：{}", key.gen());
        }

        return values;
    }

    @Override
    public List<JobPersistentObject> query(long lastFireTimeMills, int num) {
        // 每次处理一个分区的所有数据
        String lockKey = JOB_LONG_TERM_LOCK_KEY.gen();

        if (distributedLock.lock(lockKey)) {
            long index = lastFireTimeMills / properties.getFetchLongTermJobMills();
            KeyGenerator key = JOB_LONG_TERM_KEY.copy().appendKey(index);

            List<JobPersistentObject> values = redisHashMapAdapter.values(key, JobPersistentObject.class);
            redisHashMapAdapter.remove(key);
            redisZSetAdapter.removeByScore(JOB_LONG_TERM_INDEX_KEY, index, index);

            distributedLock.unlock(lockKey);
            return values;
        } else {
            if (log.isDebugEnabled()) {
                log.debug("获取分布式锁失败，不予执行");
            }
        }

        return Lists.newArrayList();
    }
}
