package com.bxm.newidea.component.payment.service.impl;

import com.alibaba.fastjson.JSON;
import com.alipay.api.AlipayApiException;
import com.alipay.api.internal.util.AlipaySignature;
import com.bxm.newidea.component.payment.config.AlipayMchAccount;
import com.bxm.newidea.component.payment.enums.PaymentActionEnum;
import com.bxm.newidea.component.payment.enums.PaymentPlatformEnum;
import com.bxm.newidea.component.payment.request.AlipanyPaymentCallbackRequest;
import com.bxm.newidea.component.payment.response.AlipayPaymentCallbackResponse;
import com.bxm.newidea.component.payment.service.IPaymentAction;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * 支付宝支付回调处理
 *
 * @author liujia
 */
@Component
@Slf4j
public class AlipayPaymentCallbackAction extends BasePaymentAction implements IPaymentAction<AlipanyPaymentCallbackRequest, AlipayPaymentCallbackResponse> {

    @Override
    public AlipayPaymentCallbackResponse exec(AlipanyPaymentCallbackRequest request) {
        //验证请求是否合法
        Map<String, String> params = convertMap(request.getRequest());

        try {
            HttpServletRequest httpRequest = request.getRequest();
            PrintWriter writer = request.getResponse().getWriter();

            String appId = params.get("app_id");
            //根据appId获取支付信息
            AlipayMchAccount alipayMchAccount = configContext.obtainAlipayAccount(appId);

            boolean checkResult = AlipaySignature.rsaCheckV1(params,
                    alipayMchAccount.getPublicKey(),
                    alipayMchAccount.getCharset(),
                    alipayMchAccount.getSignType());

            if (alipayMchAccount.getEnableCheck() && !checkResult) {
                log.error("支付宝验证签名失败，请求参数：{}", JSON.toJSONString(params));
                writer.print("fail");
                return null;
            }

            AlipayPaymentCallbackResponse response = new AlipayPaymentCallbackResponse();
            response.setNotifyTime(getField("notify_time", httpRequest));
            response.setOutTradeNo(getField("out_trade_no", httpRequest));
            response.setTradeNo(getField("trade_no", httpRequest));
            response.setTradeStatus(getField("trade_status", httpRequest));

            return response;
        } catch (AlipayApiException e) {
            log.error("支付宝回调签名验证失败，请求参数为：{}", JSON.toJSONString(params));
            log.error(e.getMessage(), e);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    private String getField(String fieldName, HttpServletRequest request) {
        return new String(request.getParameter(fieldName)
                .getBytes(StandardCharsets.ISO_8859_1), StandardCharsets.UTF_8);
    }

    private Map<String, String> convertMap(HttpServletRequest request) {
        Map<String, String> paramMap = Maps.newHashMap();

        for (Map.Entry<String, String[]> entry : request.getParameterMap().entrySet()) {
            paramMap.put(entry.getKey(), Joiner.on(",").join(entry.getValue()));
        }

        return paramMap;
    }

    @Override
    public PaymentPlatformEnum matchPlatform() {
        return PaymentPlatformEnum.ALIPAY;
    }

    @Override
    public PaymentActionEnum matchAction() {
        return PaymentActionEnum.PAYMENT_CALLBACK;
    }
}
