package com.bxm.newidea.component.payment.service.impl;

import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.AlipayResponse;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.bxm.newidea.component.payment.enums.PaymentActionEnum;
import com.bxm.newidea.component.payment.enums.PaymentModeEnum;
import com.bxm.newidea.component.payment.enums.PaymentPlatformEnum;
import com.bxm.newidea.component.payment.request.AlipayH5PreOrderRequest;
import com.bxm.newidea.component.payment.response.AlipayH5PreOrderResponse;
import com.bxm.newidea.component.payment.service.IPaymentAction;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * 支付宝H5预支付订单处理
 *
 * @author liujia
 */
@Component
@Slf4j
public class AlipayH5PreOrderAction extends BasePaymentAction implements IPaymentAction<AlipayH5PreOrderRequest, AlipayH5PreOrderResponse> {

    @Override
    public AlipayH5PreOrderResponse exec(AlipayH5PreOrderRequest request) {
        if (log.isDebugEnabled()) {
            log.debug("发起支付宝预支付请求，请求参数：{}", request);
        }

        AlipayClient alipayClient = configContext.obtainAlipayClient(request.getScene());

        if (alipayClient == null) {
            return null;
        }

        AlipayTradeWapPayRequest alipayRequest = new AlipayTradeWapPayRequest();
        alipayRequest.setBizModel(createWapPayModel(request));
        alipayRequest.setNotifyUrl(request.getNotifyUrl());

        if (isNotBlank(request.getReturnUrl())) {
            String returnUrl = request.getReturnUrl() +
                    "&paymentNum=" +
                    "paymentOrderDetail.getPaymentNum()";

            alipayRequest.setReturnUrl(returnUrl);
        }

        try {
            AlipayResponse alipayResponse = alipayClient.execute(alipayRequest);

            AlipayH5PreOrderResponse response = new AlipayH5PreOrderResponse();
            response.setLink(alipayResponse.getBody());

            return response;
        } catch (AlipayApiException e) {
            log.error("支付宝发起预支付失败，请求参数为：{}", request);
            log.error(e.getMessage(), e);
        }

        return null;
    }

    private AlipayTradeWapPayModel createWapPayModel(AlipayH5PreOrderRequest request) {
        AlipayTradeWapPayModel model = new AlipayTradeWapPayModel();
        model.setOutTradeNo(request.getOrderNum());
        model.setTotalAmount(request.getAmount().toString());
        model.setSubject(request.getOrderTitle());
        model.setBody(request.getOrderDescription());
        model.setProductCode("QUICK_WAP_WAY");
        model.setTimeoutExpress(request.getTimeoutExpress());
        return model;
    }

    @Override
    public PaymentPlatformEnum matchPlatform() {
        return PaymentPlatformEnum.ALIPAY;
    }

    @Override
    public PaymentActionEnum matchAction() {
        return PaymentActionEnum.PRE_PAYMENT;
    }

    @Override
    public PaymentModeEnum matchMode() {
        return PaymentModeEnum.H5;
    }
}
