package com.bxm.newidea.component.quartz;

import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.vo.Message;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.quartz.QuartzJobBean;

import java.util.Map;

/**
 * 定时任务调度中心添加新的定时任务需要继承该类
 * @author liujia
 */
public abstract class AbstractCustomJob extends QuartzJobBean {

    private Logger logger = LoggerFactory.getLogger(AbstractCustomJob.class);

    /**
     * 任务名称
     */
    private String jobName;

    /**
     * 任务执行周期（使用cron表达式）
     */
    private String cron;

    /**
     * 任务分组，同一个分组的任务可以进行统一调配
     */
    private String group;

    /**
     * 定时任务的业务内容，方便页面管理时了解定时器并进行管理
     */
    private String description;

    /**
     * 任务优先级，任务运行时间冲突时，优先调度的任务
     */
    private int priority;

    /**
     * 任务运行时需要的参数信息
     */
    private Map<String, Object> param;

    public AbstractCustomJob(String jobName, String cron, String description) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = "default";
        this.description = description;
    }

    public AbstractCustomJob(String jobName, String group, String cron, String description) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = group;
        this.description = description;
    }

    public AbstractCustomJob(String jobName, String group, String cron, String description, int priority, Map<String, Object> param) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = group;
        this.description = description;
        this.priority = priority;
        this.param = param;
    }

    public String getJobName() {
        return jobName;
    }

    public String getCron() {
        return cron;
    }

    public String getGroup() {
        return group;
    }

    public String getDescription() {
        return description;
    }

    public int getPriority() {
        return priority;
    }

    public Map<String, Object> getParam() {
        return param;
    }

    @Override
    protected void executeInternal(JobExecutionContext context) {
        try {
            Message result = this.service();
            if (!result.isSuccess()) {
                this.saveLog(result);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            this.saveLog(Message.build().setMessage(e.getMessage()).setSuccess(false));
        }
    }

    /**
     * 实现此方法执行具体的业务方法
     * @return message
     */
    protected abstract Message service();

    /**
     * 记录执行日志
     * @param errorMsg 异常信息
     */
    protected void saveLog(Message errorMsg) {
        logger.error(LogMarker.BIZ, "quartz error:[{}]", errorMsg);
    }
}
