package com.bxm.newidea.component.quartz;

import java.util.Date;
import java.util.Map;

import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.quartz.QuartzJobBean;

/**
 * 定时任务调度中心添加新的定时任务需要继承该类
 * 子类必须提供一个空的构造函数
 *
 * @author liujia
 */
public abstract class AbstractCustomJob extends QuartzJobBean {

    protected Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 任务名称
     */
    private String jobName;

    /**
     * 任务执行周期（使用cron表达式）
     */
    private String cron;

    /**
     * 任务分组，同一个分组的任务可以进行统一调配
     */
    private String group;

    /**
     * 定时任务的业务内容，方便页面管理时了解定时器并进行管理
     */
    private String description;

    /**
     * 任务优先级，任务运行时间冲突时，优先调度的任务
     */
    private int priority;

    /**
     * 任务运行总次数，不设置则为无限.仅simple job有效
     */
    private int count = -1;

    /**
     * 任务开始运行时间，不设置则为当前时间立即开始运行
     */
    private Date startTime;

    /**
     * 任务运行时需要的参数信息
     */
    private Map<String, Object> param;

    public AbstractCustomJob(String jobName, String cron, String description) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = "default";
        this.description = description;
    }

    public AbstractCustomJob(String jobName, String group, String cron, String description) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = group;
        this.description = description;
    }

    public AbstractCustomJob(String jobName, String group) {
        this.jobName = jobName;
        this.group = group;
    }

    public AbstractCustomJob(String jobName, String group, String cron, String description, int priority, Map<String, Object> param) {
        this.jobName = jobName;
        this.cron = cron;
        this.group = group;
        this.description = description;
        this.priority = priority;
        this.param = param;
    }

    public int getCount() {
        return count;
    }

    public void setCount(int count) {
        this.count = count;
    }

    public Date getStartTime() {
        return startTime;
    }

    public void setStartTime(Date startTime) {
        this.startTime = startTime;
    }

    public String getJobName() {
        return this.jobName;
    }

    public void setJobName(String jobName) {
        this.jobName = jobName;
    }

    public String getCron() {
        return this.cron;
    }

    public void setCron(String cron) {
        this.cron = cron;
    }

    public String getGroup() {
        return this.group;
    }

    public void setGroup(String group) {
        this.group = group;
    }

    public String getDescription() {
        return this.description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public int getPriority() {
        return this.priority;
    }

    public void setPriority(int priority) {
        this.priority = priority;
    }

    public void addParam(String key, Object value) {
        if (this.param == null) {
            this.param = Maps.newHashMap();
        }
        this.param.put(key, value);
    }

    public Map<String, Object> getParam() {
        return this.param;
    }

    public void setParam(Map<String, Object> param) {
        this.param = param;
    }

    @Override
    protected void executeInternal(JobExecutionContext context) {
        logger.debug("当前执行任务类：{}, 执行开始时间：{}", this.getClass(), DateUtils.formatDateTime(new Date()));

        try {
            Message result = this.service();
            if (!result.isSuccess()) {
                this.saveLog(result);
            }
        } catch (Exception e) {
            this.logger.error(e.getMessage(), e);
            this.saveLog(Message.build().setMessage(e.getMessage()).setSuccess(false));
        }

        logger.debug("执行结束时间：{}", DateUtils.formatDateTime(new Date()));
    }

    /**
     * 实现此方法执行具体的业务方法
     *
     * @return message
     */
    protected abstract Message service();

    /**
     * 记录执行日志
     *
     * @param errorMsg 异常信息
     */
    protected void saveLog(Message errorMsg) {
        this.logger.error("quartz error:[{}]", errorMsg);
    }

}
