package com.bxm.newidea.component.quartz.service.impl;

import com.bxm.newidea.component.quartz.AbstractCustomJob;
import com.bxm.newidea.component.quartz.domain.ScheduleViewMapper;
import com.bxm.newidea.component.quartz.service.ScheduleJobService;
import com.bxm.newidea.component.quartz.vo.ScheduleJobBean;
import com.bxm.newidea.component.quartz.vo.ScheduleJobPageParam;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import org.apache.commons.collections.CollectionUtils;
import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Map;

/**
 * 定时任务接口实现类包括定时任务初始化，更新，暂停，立即运行，删除
 */
@Service
public class ScheduleJobServiceImpl implements InitializingBean, ApplicationContextAware, ScheduleJobService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired(required = false)
    private ScheduleViewMapper scheduleViewMapper;

    private Scheduler scheduler;

    private ApplicationContext applicationContext;

    /**
     * 初始化定时任务判断定时任务是否已经存在，如果不存在则添加到调度中心
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        this.scheduler = this.applicationContext.getBean(Scheduler.class);
        Map<String, AbstractCustomJob> jobs = this.applicationContext.getBeansOfType(AbstractCustomJob.class);
        if (jobs.size() > 0) {
            JobKey key;
            for (AbstractCustomJob job : jobs.values()) {
                key = new JobKey(job.getJobName(), job.getGroup());
                if (StringUtils.isNotBlank(job.getCron())) {
                    if (!CronExpression.isValidExpression(job.getCron())) {
                        this.logger.error("jbo:[{},{}],cron表达式[{}]错误", job.getJobName(), job.getGroup(), job.getCron());
                        break;
                    }
                    if (!this.scheduler.checkExists(key)) {
                        JobDetail jobDetail = JobBuilder.newJob(job.getClass()).withIdentity(job.getJobName(), job.getGroup()).build();

                        CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(job.getCron());

                        CronTrigger cronTrigger = TriggerBuilder.newTrigger()
                                .withIdentity(job.getJobName(), job.getGroup())
                                .withPriority(job.getPriority())
                                .withSchedule(scheduleBuilder).withDescription(job.getDescription())
                                .build();

                        if (job.getParam() != null) {
                            jobDetail.getJobDataMap().putAll(job.getParam());
                        }

                        this.scheduler.scheduleJob(jobDetail, cronTrigger);
                    }
                } else {
                    logger.info("{}-{}不是一个cron定时任务", job.getJobName(), job.getGroup());
                }
            }
        }
    }

    @Override
    public Message addSimpleJob(AbstractCustomJob job) {
        Preconditions.checkArgument(job != null);

        JobBuilder detail = JobBuilder.newJob(job.getClass())
                .withIdentity(job.getJobName(), job.getGroup())
                .withDescription(job.getDescription());

        if (null != job.getParam()) {
            detail.setJobData(new JobDataMap(job.getParam()));
        }

        SimpleScheduleBuilder scheduleBuilder = SimpleScheduleBuilder.simpleSchedule()
                .withRepeatCount(job.getCount())
                .withIntervalInSeconds(1)
                .withMisfireHandlingInstructionFireNow();

        Trigger trigger = TriggerBuilder.newTrigger().withSchedule(scheduleBuilder)
                .withPriority(job.getPriority())
                .startAt(job.getStartTime())
                .build();

        Message message = Message.build(true);

        try {
            this.scheduler.scheduleJob(detail.build(), trigger);
        } catch (SchedulerException e) {
            logger.error(e.getMessage(), e);
            message.setMessage(e.getMessage()).setSuccess(false);
        }

        return message;
    }

    @Override
    public Message addCronJob(AbstractCustomJob job) {
        Preconditions.checkArgument(job != null);
        Preconditions.checkArgument(job.getCron() != null);

        JobBuilder detail = JobBuilder.newJob(job.getClass())
                .withIdentity(job.getJobName(), job.getGroup())
                .withDescription(job.getDescription());

        if (null != job.getParam()) {
            detail.setJobData(new JobDataMap(job.getParam()));
        }

        CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(job.getCron());

        Trigger trigger = TriggerBuilder.newTrigger().withSchedule(scheduleBuilder)
                .withPriority(job.getPriority())
                .build();

        Message message = Message.build(true);

        try {
            this.scheduler.scheduleJob(detail.build(), trigger);
        } catch (SchedulerException e) {
            logger.error(e.getMessage(), e);
            message.setMessage(e.getMessage()).setSuccess(false);
        }
        return message;
    }

    @Override
    public Message update(ScheduleJobBean param) {
        ScheduleJobBean scheduleJobBean = this.getJob(param.getJobName(), param.getJobGroup());
        scheduleJobBean.setCronExpression(param.getCronExpression());
        scheduleJobBean.setDescription(param.getDescription());

        try {
            TriggerKey triggerKey = this.getTriggerKey(scheduleJobBean.getJobName(), scheduleJobBean.getJobGroup());

            if (!CronExpression.isValidExpression(scheduleJobBean.getCronExpression())) {
                return Message.build(false, "cron表达式不正确");
            }

            //表达式调度构建器
            CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(scheduleJobBean.getCronExpression());

            CronTrigger cronTrigger = this.getCronTrigger(this.scheduler, scheduleJobBean.getJobName(), scheduleJobBean.getJobGroup());

            assert cronTrigger != null;
            cronTrigger = cronTrigger.getTriggerBuilder()
                    .withIdentity(triggerKey)
                    .withSchedule(scheduleBuilder)
                    .withPriority(scheduleJobBean.getPriority())
                    .withDescription(scheduleJobBean.getDescription())
                    .build();

            this.scheduler.rescheduleJob(triggerKey, cronTrigger);

            Trigger.TriggerState triggerState = this.scheduler.getTriggerState(triggerKey);

            if (scheduleJobBean.getParam() != null && scheduleJobBean.getParam().size() > 0) {
                cronTrigger.getJobDataMap().putAll(scheduleJobBean.getParam());
            }

            if (Trigger.TriggerState.PAUSED.name().equals(triggerState.name())) {
                this.pause(scheduleJobBean.getJobName(), scheduleJobBean.getJobGroup());
            }

            return Message.build(true);

        } catch (SchedulerException e) {
            this.logger.error("更新定时任务失败", e);
            return Message.build(false, "更新定时任务失败");
        }
    }

    @Override
    public Message run(String jobName, String jobGroup) {
        JobKey jobKey = this.getJobKey(jobName, jobGroup);
        try {
            this.scheduler.triggerJob(jobKey);
            return Message.build(true);
        } catch (SchedulerException e) {
            this.logger.error("运行定时任务失败", e);
            return Message.build(false, "运行定时任务失败");
        }

    }

    @Override
    public Message resume(String jobName, String jobGroup) {
        JobKey jobKey = this.getJobKey(jobName, jobGroup);
        try {
            this.scheduler.resumeJob(jobKey);
            return Message.build(true);
        } catch (SchedulerException e) {
            this.logger.error("恢复定时任务失败", e);
            return Message.build(false, "恢复定时任务失败");
        }
    }

    @Override
    public Message pause(String jobName, String jobGroup) {
        JobKey jobKey = this.getJobKey(jobName, jobGroup);
        try {
            this.scheduler.pauseJob(jobKey);
            return Message.build(true);
        } catch (SchedulerException e) {
            this.logger.error("暂停定时任务失败");
            return Message.build(false, "暂停定时任务失败");
        }
    }

    @Override
    public Message remove(String jobName, String jobGroup) {
        JobKey jobKey = this.getJobKey(jobName, jobGroup);
        try {
            this.scheduler.deleteJob(jobKey);
            return Message.build(true);
        } catch (SchedulerException e) {
            this.logger.error("删除定时任务失败");
            return Message.build(false, "删除定时任务失败");
        }
    }

    @Override
    public PageWarper<ScheduleJobBean> queryJobs(ScheduleJobPageParam scheduleJobPageParam) {
        if (null == scheduleViewMapper) {
            return null;
        }
        PageWarper<ScheduleJobBean> scheduleJobBeanPageWarper = new PageWarper<>(this.scheduleViewMapper.selectByPageNumSize(scheduleJobPageParam));
        try {
            if (CollectionUtils.isNotEmpty(scheduleJobBeanPageWarper.getList())) {
                for (ScheduleJobBean item : scheduleJobBeanPageWarper.getList()) {
                    // 获取任务状态
                    Trigger.TriggerState state = this.scheduler.getTriggerState(this.getTriggerKey(item.getJobName(), item.getJobGroup()));
                    item.setStatus(state.name());
                    item.setPrevTime(new Date(item.getPrevFireTime()));
                    item.setNextTime(new Date(item.getNextFireTime()));
                }
            }
        } catch (SchedulerException e) {
            this.logger.error("获取定时任务状态失败", e);
        }
        return scheduleJobBeanPageWarper;
    }

    @Override
    public ScheduleJobBean getJob(String jobName, String jobGroup) {
        if (null == scheduleViewMapper) {
            return null;
        }
        return this.scheduleViewMapper.selectByJobNameAndJobGroup(jobName, jobGroup);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    /**
     * 获取触发器的key值
     * @param triggerGroup 触发器名称
     * @param triggerGroup 触发器分组
     * @return 触发器
     */
    private TriggerKey getTriggerKey(String triggerName, String triggerGroup) {
        return TriggerKey.triggerKey(triggerName, triggerGroup);
    }

    /**
     * 获取任务的key值
     * @param jobName  任务名称
     * @param jobGroup 任务分组
     * @return 任务key
     */
    private JobKey getJobKey(String jobName, String jobGroup) {
        return JobKey.jobKey(jobName, jobGroup);
    }

    /**
     * 获取cron表达式触发器
     * @param jobName   任务名称
     * @param jobGroup  任务分组
     * @param scheduler 调度对象
     * @return cron触发器
     */
    private CronTrigger getCronTrigger(Scheduler scheduler, String jobName, String jobGroup) {
        try {
            return (CronTrigger) scheduler.getTrigger(this.getTriggerKey(jobName, jobGroup));
        } catch (SchedulerException e) {
            this.logger.error("获取cron表达式触发器失败", e);
            return null;
        }
    }

}
