package com.bxm.newidea.component.redis;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * redis 散列操作代理类
 * @author liujia 2018/5/21 18:16
 */
public interface RedisHashMapAdapter extends BaseRedisOperation {

    /**
     * 添加值到散列
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     */
    void put(KeyGenerator generator, String subKey, Object value);

    /**
     * 在散列中存放用于自增的存储值，这个值将不会进行序列号
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     */
    void putLong(KeyGenerator generator, String subKey, Long value);

    /**
     * 批量添加值到散列
     * @param generator 散列key
     * @param values    散列中存储的值
     */
    void putAll(KeyGenerator generator, Map<String, ? extends Object> values);

    /**
     * 从散列中获取值
     * @param generator 散列键名
     * @param subKey    散列存储键名
     * @return 散列中指定键名存储的类型
     */
    <T> T get(KeyGenerator generator, String subKey);

    /**
     * 从散列中获取未经序列化的数值（此值必须通过{@link #putLong(KeyGenerator, String, Long)}或{@link #increment(KeyGenerator, String, int)}加入）
     * @param generator 散列键名
     * @param subKey    散列存储键名
     * @return 散列中指定键名存储的值
     */
    Long getLong(KeyGenerator generator, String subKey);

    /**
     * @param key    键值
     * @param subKey 散列存储键名
     * @param <T>    返回类型
     * @return 返回的结果
     */
    <T> T redisGet(String key, String subKey);

    /**
     * 获取当前散列中包含的全部键值
     * @param generator key
     * @return 键列表
     */
    Set<String> keys(KeyGenerator generator);

    /**
     * 获取当前散列中存储的所有值
     * @param generator key
     * @return 值列表
     */
    <T> List<T> values(KeyGenerator generator);

    /**
     * 返回当前散列的全部数据,方法使用scan实现，可能会存在脏数据的可能性
     * @param generator key
     * @return 指定散列的全部数据
     */
    <T> Map<String, T> entries(KeyGenerator generator);

    /**
     * 判断散列中是否存在对应的键
     * @param generator 散列键名
     * @param subKey    存储键
     * @return true表示存在
     */
    boolean exists(KeyGenerator generator, String subKey);

    /**
     * 根据键移除散列中的存储信息
     * @param generator 散列键
     * @param subKeys   存储键
     * @return 移除的数量
     */
    long remove(KeyGenerator generator, String... subKeys);

    /**
     * 获取当前散列中的存储键数量
     * @param generator 散列键
     * @return 存储的key数量
     */
    long length(KeyGenerator generator);

    /**
     * 对subkey对应的值进行自增操作，如果对应的值不存在则默认进行创建，默认为1
     * @param generator 散列键
     * @param subKey    存储键
     * @param amount    自增步长
     * @return 自增后的结果
     */
    long increment(KeyGenerator generator, String subKey, int amount);
}
