package com.bxm.newidea.component.redis;

import org.springframework.data.redis.core.SetOperations;

import java.util.Set;

/**
 * redis集合类操作代理
 * set类型如果存放对象，请重新equals和hashcode方法
 * @author liujia 2018/5/22 10:12
 */
public interface RedisSetAdapter extends BaseRedisOperation {

    /**
     * 添加到集合中
     * @param generator key
     * @param values    添加到集合中的数据（如果存在重复则会被忽略）
     * @return 添加的数量
     */
    long add(KeyGenerator generator, Object... values);

    /**
     * 从集合中移除指定的值
     * @param generator key
     * @param values 待移除的数据
     * @return 移除的数量
     */
    long remove(KeyGenerator generator,Object... values);

    /**
     * 判断集合中是否存在指定的值
     * @param generator key
     * @param value     检测的值
     * @return true表示存在
     */
    boolean exists(KeyGenerator generator, Object value);

    /**
     * 集合中包含元素的数量
     * @param generator key
     * @return key包含元素的数量
     */
    long size(KeyGenerator generator);

    /**
     * 获取集合中全部的成员
     * @param generator key
     * @return 集合中的全部成员，通过游标获取，有可能会返回重复的、已删除
     */
    <T> Set<T> getAllMembers(KeyGenerator generator);

    /**
     * 返回存在第一个键值中但是不存在于其他集合的数据（差集计算）
     * @param generator       主集合
     * @param otherGenerators 用于比较的集合
     * @return 差集结果
     */
    <T> Set<T> difference(KeyGenerator generator,KeyGenerator... otherGenerators);

    /**
     * 计算存在于<code>generator</code>，但是不存在于<code>otherGenerators</code>的差集结果并将结果保存到
     * <code>storeGenerator</code>中
     * @param storeGenerator  存储的key
     * @param generator       原始数据的key
     * @param otherGenerators 用于比较的keys
     * @return 结果数量
     */
    long differenceAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators);

    /**
     * 获取<code>generator</code>和<code>otherGenerators</code>的数据交集
     * @param generator       主集合
     * @param otherGenerators 用于比较的集合
     * @return 交集结果
     */
    <T> Set<T> inter(KeyGenerator generator, KeyGenerator... otherGenerators);

    /**
     * 计算存在于<code>generator</code>，但是不存在于<code>otherGenerators</code>的交集结果并将结果保存到
     * <code>storeGenerator</code>中
     * @param storeGenerator  存储的key
     * @param generator       计算交集的主集合
     * @param otherGenerators 和主集合比较的其他集合
     * @return 交集结果数量
     */
    long interAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators);

    /**
     * 获取<code>generator</code>和<code>otherGenerators</code>的数据并集
     * @param generator       主集合
     * @param otherGenerators 用于比较的集合
     * @return 并集结果
     */
    <T> Set<T> union(KeyGenerator generator, KeyGenerator... otherGenerators);

    /**
     * 计算存在于<code>generator</code>，但是不存在于<code>otherGenerators</code>的并集结果并将结果保存到
     * <code>storeGenerator</code>中
     * @param storeGenerator  存储的key
     * @param generator       计算交集的主集合
     * @param otherGenerators 和主集合比较的其他集合
     * @return 并集结果数量
     */
    long unionAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators);

    /**
     * 从<code>sourceKey</code>中判断是否存在<code>item</code>，如果存在则将item移动到<code>targetKey</code>
     * @param sourceKey 原始集合
     * @param targetKey 目标集合
     * @param item      移动的目标
     * @return true表示移动成功
     */
    boolean move(KeyGenerator sourceKey, KeyGenerator targetKey, Object item);

    /**
     * 获取代理操作的元素对象
     * @return set操作类
     */
    SetOperations getOriginal();
}
