package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bxm.newidea.component.tools.NumberUtils.parseToLong;

@SuppressWarnings("unchecked")
@Component
public class RedisHashMapAdapterImpl extends BaseRedisAdapter implements RedisHashMapAdapter {

    private HashOperations<String, String, Object> hashOperations;

    @Autowired
    public RedisHashMapAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        this.hashOperations = redisTemplate.opsForHash();
    }

    @Override
    public void put(KeyGenerator generator, String subKey, Object value) {
        Assert.notNull(generator, "key构建器必须传递");

        this.hashOperations.put(generator.gen(), subKey, this.serializer.serialize(value));
    }

    @Override
    public void putLong(KeyGenerator generator, String subKey, Long value) {
        if (generator == null || null == value) {
            return;
        }
        this.hashOperations.put(generator.gen(), subKey, this.stringSerializer.serialize(value.toString()));
    }

    @Override
    public void putAll(KeyGenerator generator, Map<String, ? extends Object> values) {
        if (values == null) {
            return;
        }

        Map<String, byte[]> convertResult = new HashMap<>();

        values.forEach((key, value) -> {
            convertResult.put(key, this.serializer.serialize(value));
        });

        this.hashOperations.putAll(generator.gen(), convertResult);
    }

    @Override
    public <T> T get(KeyGenerator generator, String subKey) {
        String key = generator.gen();
        Object value = this.hashOperations.get(key, subKey);
        if (null == value) {
            return null;
        }

        return this.deserialize(value);
    }

    @Override
    public Long getLong(KeyGenerator generator, String subKey) {
        Object value = this.hashOperations.get(generator.gen(), subKey);
        if (null == value) {
            return 0L;
        }
        return parseToLong(this.stringSerializer.deserialize((byte[]) value));
    }

    @Override
    public <T> T redisGet(String key, String subKey) {
        Object value = this.hashOperations.get(key, subKey);
        if (null == value) {
            return null;
        }

        return this.deserialize(value);
    }

    @Override
    public Set<String> keys(KeyGenerator generator) {
        return this.hashOperations.keys(generator.gen());
    }

    @Override
    public <T> List<T> values(KeyGenerator generator) {
        List result = this.hashOperations.values(generator.gen());
        return (List<T>) result.stream().map(item -> this.serializer.deserialize((byte[]) item)).collect(Collectors.toList());
    }

    @Override
    public <T> Map<String, T> entries(KeyGenerator generator) {
        ScanOptions.scanOptions().build();
        Cursor<Map.Entry<String, Object>> cursor = this.hashOperations.scan(generator.gen(), ScanOptions.scanOptions().count(10000).build());

        Map<String, T> result = new HashMap<>();
        Map.Entry<String, Object> entry = null;

        while (cursor.hasNext()) {
            entry = cursor.next();
            result.put(entry.getKey(), (T) this.serializer.deserialize((byte[]) entry.getValue()));
        }

        return result;
    }

    @Override
    public boolean exists(KeyGenerator generator, String subKey) {
        return this.hashOperations.hasKey(generator.gen(), subKey);
    }

    @Override
    public long remove(KeyGenerator generator, String... subKeys) {
        if (ArrayUtils.isEmpty(subKeys)) {
            return 0L;
        }

        return this.hashOperations.delete(generator.gen(), subKeys);
    }

    @Override
    public long length(KeyGenerator generator) {
        return this.hashOperations.size(generator.gen());
    }

    @Override
    public long increment(KeyGenerator generator, String subKey, int amount) {
        return this.hashOperations.increment(generator.gen(), subKey, amount);
    }
}
