package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.Set;


@SuppressWarnings("unchecked")
@Component
public class RedisSetAdapterImpl extends BaseRedisAdapter implements RedisSetAdapter {

    private SetOperations operations;

    @Autowired
    public RedisSetAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        operations = redisTemplate.opsForSet();
    }

    @Override
    public long add(KeyGenerator generator, Object... values) {
        String key = generator.gen();
        try {
            redisTemplate.watch(key);
            redisTemplate.multi();

            return operations.add(generator.gen(), serialize(values));
        } finally {
            redisTemplate.unwatch();
        }
    }

    @Override
    public long remove(KeyGenerator generator, Object... values) {
        return operations.remove(generator.gen(),serialize(values));
    }

    @Override
    public boolean exists(KeyGenerator generator, Object value) {
        return operations.isMember(generator.gen(), serializer.serialize(value));
    }

    @Override
    public long size(KeyGenerator generator) {
        return operations.size(generator.gen());
    }

    @Override
    public <T> Set<T> getAllMembers(KeyGenerator generator) {
        Cursor cursor = operations.scan(generator.gen(), ScanOptions.scanOptions().count(10000).build());

        Set<T> result = new HashSet<>();
        while (cursor.hasNext()){
            result.add(deserialize(cursor.next()));
        }

        return result;
    }

    private <T> Set<T> deserialize(Set<Object> data) {
        Set<T> result = new HashSet<>();

        if (null != data && data.size() > 0) {
            data.forEach(item -> result.add((T) serializer.deserialize((byte[]) item)));
        }

        return result;
    }

    @Override
    public <T> Set<T> difference(KeyGenerator generator, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.difference(generator.gen(), convertKeys(otherGenerators)));
    }

    @Override
    public long differenceAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.differenceAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public <T> Set<T> inter(KeyGenerator generator, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.intersect(generator.gen(), convertKeys(otherGenerators)));
    }

    @Override
    public long interAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.intersectAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public <T> Set<T> union(KeyGenerator generator, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.union(generator.gen(), convertKeys(otherGenerators)));
    }

    @Override
    public long unionAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.unionAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public boolean move(KeyGenerator sourceKey, KeyGenerator targetKey, Object item) {
        return operations.move(sourceKey.gen(), serializer.serialize(item), targetKey.gen());
    }

    @Override
    public SetOperations getOriginal() {
        return operations;
    }
}
