package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.DefaultTypedTuple;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Component;

import java.util.LinkedHashSet;
import java.util.Set;
import java.util.stream.Collectors;

@SuppressWarnings("unchecked")
@Component
public class RedisZSetAdapterImpl extends BaseRedisAdapter implements RedisZSetAdapter {

    private ZSetOperations operations;

    @Autowired
    public RedisZSetAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        this.operations = redisTemplate.opsForZSet();
    }

    @Override
    public ZSetOperations getOriginal() {
        return this.operations;
    }

    @Override
    public Boolean add(KeyGenerator generator, Object value, double score) {
        return this.operations.add(generator.gen(), this.serializer.serialize(value), score);
    }

    @Override
    public long add(KeyGenerator generator, Set<ZSetOperations.TypedTuple> tuples) {
        return this.operations.add(generator.gen(), this.serialize(tuples));
    }

    private Set<ZSetOperations.TypedTuple> serialize(Set<ZSetOperations.TypedTuple> tuples) {
        return tuples.stream().map(tuple -> new DefaultTypedTuple<>(this.serializer.serialize(tuple.getValue()), tuple.getScore())).collect(Collectors.toSet());
    }

    @Override
    public double incrementScore(KeyGenerator generator, Object value, double amount) {
        String key = generator.gen();
        try {
            this.redisTemplate.watch(key);
            this.redisTemplate.multi();

            return this.operations.incrementScore(key, this.serializer.serialize(value), amount);
        } finally {
            this.redisTemplate.unwatch();
        }
    }

    @Override
    public Long remove(KeyGenerator generator, Object... values) {
        return this.operations.remove(generator.gen(), this.serialize(values));
    }


    @Override
    public Long removeByRange(KeyGenerator generator, long start, long end) {
        return this.operations.removeRange(generator.gen(), start, end);
    }

    @Override
    public Long removeByScore(KeyGenerator generator, double min, double max) {
        return this.operations.removeRangeByScore(generator.gen(), min, max);
    }

    @Override
    public Long size(KeyGenerator generator) {
        return this.operations.size(generator.gen());
    }

    @Override
    public Long count(KeyGenerator generator, double min, double max) {
        return this.operations.count(generator.gen(), min, max);
    }

    @Override
    public Long rank(KeyGenerator generator, Object value) {
        return this.operations.rank(generator.gen(), this.serializer.serialize(value));
    }

    @Override
    public Double score(KeyGenerator generator, Object value) {
        return this.operations.score(generator.gen(), this.serializer.serialize(value));
    }

    @Override
    public <T> Set<ZSetOperations.TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max,
                                                                        long offset, long count, boolean isReversed) {
        Set<ZSetOperations.TypedTuple> result;
        if (isReversed) {
            result = this.operations.reverseRangeByScoreWithScores(generator.gen(), min, max, offset, count);
        } else {
            result = this.operations.rangeByScoreWithScores(generator.gen(), min, max, offset, count);
        }
        return this.deserializeTuple(result);
    }

    private <T> Set<ZSetOperations.TypedTuple<T>> deserializeTuple(Set<ZSetOperations.TypedTuple> tuples) {
        Set<ZSetOperations.TypedTuple<T>> result = new LinkedHashSet<>();

        for (ZSetOperations.TypedTuple tuple : tuples) {
            result.add(new DefaultTypedTuple<>((T) this.serializer.deserialize((byte[]) tuple.getValue()), tuple.getScore()));
        }

        return result;
    }

    @Override
    public <T> Set<ZSetOperations.TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max,
                                                                        boolean isReversed) {
        Set<ZSetOperations.TypedTuple> result;

        if (isReversed) {
            result = this.operations.reverseRangeByScoreWithScores(generator.gen(), min, max);
        } else {
            result = this.operations.rangeByScoreWithScores(generator.gen(), min, max);
        }

        return this.deserializeTuple(result);
    }

    private <T> Set<T> deserialize(Set<Object> data) {
        Set<T> result = new LinkedHashSet<>();

        for (Object obj : data) {
            result.add((T) this.serializer.deserialize((byte[]) obj));
        }

        return result;
    }

    @Override
    public <T> Set<T> range(KeyGenerator generator, long start, long end, boolean isReversed) {
        Set result;
        if (isReversed) {
            result = this.operations.reverseRange(generator.gen(), start, end);
        } else {
            result = this.operations.range(generator.gen(), start, end);
        }
        return (Set<T>) this.deserialize(result);
    }

    @Override
    public <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, boolean isReversed) {
        Set result;
        if (isReversed) {
            result = this.operations.reverseRangeByScore(generator.gen(), min, max);
        } else {
            result = this.operations.rangeByScore(generator.gen(), min, max);
        }
        return this.deserialize(result);
    }

    @Override
    public <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, long offset,
                                   long count, boolean isReversed) {
        Set result;
        if (isReversed) {
            result = this.operations.reverseRangeByScore(generator.gen(), min, max, offset, count);
        } else {
            result = this.operations.rangeByScore(generator.gen(), min, max, offset, count);
        }
        return (Set<T>) this.deserialize(result);
    }

    @Override
    public Long intersectAndStore(KeyGenerator target, KeyGenerator source, KeyGenerator... otherKeys) {
        return this.operations.intersectAndStore(source.gen(), this.convertKeys(otherKeys), target.gen());
    }

    @Override
    public Long unionAndStore(KeyGenerator target, KeyGenerator source, KeyGenerator... otherKeys) {
        return this.operations.unionAndStore(source.gen(), this.convertKeys(otherKeys), target.gen());
    }
}
