package com.bxm.newidea.component.redis;

/**
 * 字符串类型redis操作适配
 * @author liujia 2018/5/21 15:16
 */
public interface RedisStringAdapter extends BaseRedisOperation {
    /**
     * 值增加，不存在时返回<code>1</code>并设置为1
     * @param generator key创建器
     * @return 增长后的值
     */
    Long increment(KeyGenerator generator);

    /**
     * 值增加，不存在时返回<code>1</code>并设置为1
     * @param generator key创建器
     * @param amount    当前值+amount进行增加
     * @return 增长后的值
     */
    Long increment(KeyGenerator generator, int amount);

    /**
     * 值增加，不存在时返回<code>1</code>并设置为1
     * @param generator     key创建器
     * @param expireSeconds key过期的时间（单位：秒）
     * @return 增长后的值
     */
    Long increment(KeyGenerator generator, long expireSeconds);

    /**
     * 从defaultVal开始自增，每次增加1
     * @param generator  key创建器
     * @param defaultVal 起始值
     * @return 增长后的值，如果缓存不存在则返回defaultVal并进行设置
     */
    Long incrementWithDefault(KeyGenerator generator, long defaultVal);

    /**
     * 从defaultVal开始自增，每次增加amount
     * @param generator  key创建器
     * @param defaultVal 起始值
     * @param amount     增长的步长
     * @return 增长后的值，如果缓存不存在则返回defaultVal并进行设置
     */
    Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount);

    /**
     * 从defaultVal开始自增，每次增加amount，在expireSeconds秒后过期
     * @param generator     key创建器
     * @param defaultVal    起始值
     * @param amount        增长的步长
     * @param expireSeconds 过期时间
     * @return 增长后的值，如果缓存不存在则返回defaultVal并进行设置
     */
    Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount, long expireSeconds);

    /**
     * 获取值并递减，不存在时返回0并设置值为0
     * @param generator key创建器
     * @return 递减后的值
     */
    Long decrement(KeyGenerator generator);

    /**
     * 获取值并递减，不存在时返回0并设置值为0
     * @param generator key创建器
     * @param amount    递减步长
     * @return 递减后的值
     */
    Long decrement(KeyGenerator generator, int amount);

    /**
     * 获取值并递减，不存在时返回0并设置值为0
     * @param generator     key创建器
     * @param expireSeconds 过期时间（秒）
     * @return 递减后的值
     */
    Long decrement(KeyGenerator generator, long expireSeconds);

    /**
     * 从defaultVal开始递减，每次减少amount
     * @param generator  key创建器
     * @param defaultVal 起始值
     * @param amount     递减步长
     * @return 递减后的值，如果缓存不存在则返回defaultVal
     */
    Long decrement(KeyGenerator generator, long defaultVal, int amount);

    /**
     * 从defaultVal开始递减，每次减少amount，在expireSeconds后过期
     * @param generator     key创建器
     * @param defaultVal    起始值
     * @param amount        递减步长
     * @param expireSeconds 过期时间
     * @return 递减后的值，如果缓存不存在则返回defaultVal
     */
    Long decrement(KeyGenerator generator, long defaultVal, int amount, long expireSeconds);

    /**
     * 根据key获取缓存的值
     * @param generator key创建器
     * @return key对应的value
     */
    Long getLong(KeyGenerator generator);

    /**
     * 根据key获取缓存的值
     * @param generator key创建器
     * @return key对应的value
     */
    Integer getInt(KeyGenerator generator);

    /**
     * 根据key获取缓存的值
     * @param generator key创建器
     * @return key对应的value(使用字符串进行序列化的结果)
     */
    String getString(KeyGenerator generator);

    /**
     * 根据key获取缓存值
     * @param generator key
     * @param clasz     获取值的class
     * @return key对应的value(使用对象序列化的结果)
     */
    <T> T get(KeyGenerator generator, Class<T> clasz);

    /**
     * 设置缓存值，value将会进行序列化，不可调用自增、自减方法
     * @param generator key创建器
     * @param value     缓存值(使用对象序列化)
     */
    <T> void set(KeyGenerator generator, T value);

    /**
     * 设置缓存值，value将序列化为字符串保存，可进行自增、自减操作
     * @param generator key
     * @param value     缓存值
     */
    void set(KeyGenerator generator, long value);

    /**
     * 设置缓存值，在设定的时间后过期
     * @param generator     key创建器
     * @param value         缓存值
     * @param expireSeconds 过期时间（秒）
     */
    <T> void set(KeyGenerator generator, T value, long expireSeconds);
}
