package com.bxm.newidea.component.redis;

import com.fasterxml.jackson.core.type.TypeReference;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * redis 散列操作代理类
 * @author liujia 2018/5/21 18:16
 */
public interface RedisHashMapAdapter extends BaseRedisOperation {

    /**
     * 添加值到散列
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     */
    <T> void put(KeyGenerator generator, String subKey, T value);

    /**
     * 在散列中存放用于自增的存储值，这个值将不会进行序列号
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     */
    void putLong(KeyGenerator generator, String subKey, Long value);

    /**
     * 批量添加值到散列
     * @param generator 散列key
     * @param values    散列中存储的值
     */
    <T> void putAll(KeyGenerator generator, Map<String, T> values);

    /**
     * 从散列中获取值
     * @param generator 散列键名
     * @param subKey    散列存储键名
     * @return 散列中指定键名存储的类型
     */
    <T> T get(KeyGenerator generator, String subKey, Class<T> clasz);

    /**
     * 从散列中获取泛型数据,如泛型集合、泛型Map
     * eg:
     * redisHashMapAdapter.get(key,subKey,new TypeReference<List<JavaBean>>{})
     * redisHashMapAdapter.get(key,subKey,new TypeReference<Map<String,JavaBean>>{})
     * @param generator 散列键名
     * @param subKey    散列存储键名
     * @return 散列中指定键名存储的类型
     */
    <T> T get(KeyGenerator generator, String subKey, TypeReference<T> typeReference);

    /**
     * 从散列中获取未经序列化的数值（此值必须通过{@link #putLong(KeyGenerator, String, Long)}或{@link #increment(KeyGenerator, String, int)}加入）
     * @param generator 散列键名
     * @param subKey    散列存储键名
     * @return 散列中指定键名存储的值
     */
    Long getLong(KeyGenerator generator, String subKey);

    /**
     * 获取当前散列中包含的全部键值
     * @param generator key
     * @return 键列表
     */
    Set<String> keys(KeyGenerator generator);

    /**
     * 获取当前散列中存储的所有值
     * @param generator key
     * @return 值列表
     */
    <T> List<T> values(KeyGenerator generator, Class<T> clasz);

    /**
     * 获取当前散列中存储的所有值，值类型为泛型数据,如泛型集合、泛型Map
     * @param generator key
     * @return 值列表
     */
    <T> List<T> values(KeyGenerator generator, TypeReference<T> typeReference);

    /**
     * 返回当前散列的全部数据,方法使用scan实现，可能会存在脏数据的可能性
     * @param generator key
     * @return 指定散列的全部数据
     */
    <T> Map<String, T> entries(KeyGenerator generator, Class<T> clasz);

    /**
     * 返回散列中的全部数据，value值为泛型的情况下使用
     * @param generator     key
     * @param typeReference 泛型类型定义
     * @return 指定散列的全部数据
     */
    <T> Map<String, T> entries(KeyGenerator generator, TypeReference<T> typeReference);

    /**
     * 判断散列中是否存在对应的键
     * @param generator 散列键名
     * @param subKey    存储键
     * @return true表示存在
     */
    Boolean exists(KeyGenerator generator, String subKey);

    /**
     * 根据键移除散列中的存储信息
     * @param generator 散列键
     * @param subKeys   存储键
     * @return 移除的数量
     */
    Long remove(KeyGenerator generator, String... subKeys);

    /**
     * 获取当前散列中的存储键数量
     * @param generator 散列键
     * @return 存储的key数量
     */
    Long length(KeyGenerator generator);

    /**
     * 对subkey对应的值进行自增操作，如果对应的值不存在则默认进行创建，默认为1
     * @param generator 散列键
     * @param subKey    存储键
     * @param amount    自增步长
     * @return 自增后的结果
     */
    Long increment(KeyGenerator generator, String subKey, int amount);

}
