package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.newidea.component.tools.NumberUtils.parseToLong;

@Component
public class RedisHashMapAdapterImpl extends BaseRedisAdapter implements RedisHashMapAdapter {

    private HashOperations<String, String, Object> hashOperations;

    @Autowired
    public RedisHashMapAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        this.hashOperations = redisTemplate.opsForHash();
    }

    @Override
    public <T> void put(KeyGenerator generator, String subKey, T value) {
        Assert.notNull(generator, "key构建器必须传递");

        this.hashOperations.put(generator.gen(), subKey, Objects.requireNonNull(this.getSerializerWithoutType().serialize(value)));
    }

    @Override
    public void putLong(KeyGenerator generator, String subKey, Long value) {
        if (generator == null || null == value) {
            return;
        }
        this.hashOperations.put(generator.gen(), subKey, Objects.requireNonNull(this.stringSerializer.serialize(value.toString())));
    }

    @Override
    public <T> void putAll(KeyGenerator generator, Map<String, T> values) {
        if (values == null) {
            return;
        }

        Map<String, byte[]> convertResult = Maps.newHashMap();
        RedisSerializer serializer = this.getSerializerWithoutType();

        values.forEach((key, value) -> convertResult.put(key, serializer.serialize(value)));

        this.hashOperations.putAll(generator.gen(), convertResult);
    }

    @Override
    public <T> T get(KeyGenerator generator, String subKey, Class<T> clasz) {
        String key = generator.gen();
        Object value = this.hashOperations.get(key, subKey);
        if (null == value) {
            return null;
        }

        return this.deserialize(value, clasz);
    }

    @Override
    public <T> T get(KeyGenerator generator, String subKey, TypeReference<T> typeReference) {
        String key = generator.gen();
        Object value = this.hashOperations.get(key, subKey);
        if (null == value) {
            return null;
        }

        return (T) getSerializer(typeReference).deserialize((byte[]) value);
    }

    @Override
    public Long getLong(KeyGenerator generator, String subKey) {
        Object value = this.hashOperations.get(generator.gen(), subKey);
        if (null == value) {
            return 0L;
        }
        return parseToLong(this.stringSerializer.deserialize((byte[]) value));
    }

    @Override
    public Set<String> keys(KeyGenerator generator) {
        return this.hashOperations.keys(generator.gen());
    }

    @Override
    public <T> List<T> values(KeyGenerator generator, Class<T> clasz) {
        List result = this.hashOperations.values(generator.gen());
        RedisSerializer<T> serializer = this.getSerializer(clasz);
        return (List<T>) result.stream().map(item -> serializer.deserialize((byte[]) item)).collect(Collectors.toList());
    }

    @Override
    public <T> List<T> values(KeyGenerator generator, TypeReference<T> typeReference) {
        List result = this.hashOperations.values(generator.gen());

        RedisSerializer<T> serializer = this.getSerializer(typeReference);
        return (List<T>) result.stream().map(item -> serializer.deserialize((byte[]) item)).collect(Collectors.toList());
    }

    @Override
    public <T> List<T> multiGet(KeyGenerator generator, Collection<String> subKeys, Class<T> clasz) {
        List result = hashOperations.multiGet(generator.gen(), subKeys);
        RedisSerializer<T> serializer = this.getSerializer(clasz);
        return (List<T>) result.stream()
                .filter(item -> null != item)
                .map(item -> serializer.deserialize((byte[]) item))
                .collect(Collectors.toList());
    }

    @Override
    public <T> Map<String, T> entries(KeyGenerator generator, Class<T> clasz) {
        return getEntries(generator, clasz, null);
    }

    @Override
    public <T> Map<String, T> entries(KeyGenerator generator, TypeReference<T> typeReference) {
        return getEntries(generator, null, typeReference);
    }

    private <T> Map<String, T> getEntries(KeyGenerator generator, Class<T> clasz, TypeReference<T> typeReference) {
        Cursor<Map.Entry<String, Object>> cursor = this.hashOperations.scan(generator.gen(), ScanOptions.scanOptions().count(10000).build());

        Map<String, T> result = new HashMap<>();
        Map.Entry<String, Object> entry;
        RedisSerializer<T> serializer = null == clasz ? this.getSerializer(typeReference) : this.getSerializer(clasz);

        while (cursor.hasNext()) {
            entry = cursor.next();
            result.put(entry.getKey(), (T) serializer.deserialize((byte[]) entry.getValue()));
        }

        return result;
    }

    @Override
    public Boolean exists(KeyGenerator generator, String subKey) {
        return this.hashOperations.hasKey(generator.gen(), subKey);
    }

    @Override
    public Long remove(KeyGenerator generator, String... subKeys) {
        if (ArrayUtils.isEmpty(subKeys)) {
            return 0L;
        }

        return this.hashOperations.delete(generator.gen(), subKeys);
    }

    @Override
    public Long length(KeyGenerator generator) {
        return this.hashOperations.size(generator.gen());
    }

    @Override
    public Long increment(KeyGenerator generator, String subKey, int amount) {
        return this.hashOperations.increment(generator.gen(), subKey, amount);
    }

}
