package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

@SuppressWarnings("unchecked")
@Component
public class RedisSetAdapterImpl extends BaseRedisAdapter implements RedisSetAdapter {

    private SetOperations operations;

    @Autowired
    public RedisSetAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        operations = redisTemplate.opsForSet();
    }

    @Override
    public Long add(KeyGenerator generator, Object... values) {
        String key = generator.gen();
        try {
            redisTemplate.watch(key);
            redisTemplate.multi();

            return operations.add(generator.gen(), serializeWithoutType(values));
        } finally {
            redisTemplate.unwatch();
        }
    }

    @Override
    public <T> Long remove(KeyGenerator generator, Object... values) {
        return operations.remove(generator.gen(), serialize(values));
    }

    @Override
    public <T> Boolean exists(KeyGenerator generator, T value) {
        return operations.isMember(generator.gen(), Objects.requireNonNull(getSerializerWithoutType().serialize(value)));
    }

    @Override
    public <T> List<T> pop(KeyGenerator generator, Long count, Class<T> clasz) {
        RedisSerializer<T> serializer = getSerializer(clasz);
        return innerPop(generator, count, serializer);
    }

    @Override
    public <T> List<T> pop(KeyGenerator generator, Long count, TypeReference<T> typeReference) {
        RedisSerializer<T> serializer = getSerializer(typeReference);
        return innerPop(generator, count, serializer);
    }

    private <T> List<T> innerPop(KeyGenerator generator, Long count, RedisSerializer<T> redisSerializer) {
        List<byte[]> result = operations.pop(generator.gen(), count);
        List<T> finalResult = Lists.newArrayList();
        if (result != null) {
            finalResult = result.stream().map(redisSerializer::deserialize).collect(Collectors.toList());
        }
        return finalResult;
    }

    @Override
    public Long size(KeyGenerator generator) {
        return operations.size(generator.gen());
    }

    @Override
    public <T> Set<T> getAllMembers(KeyGenerator generator, Class<T> clasz) {
        RedisSerializer<T> serializer = getSerializer(clasz);
        return innerGetAllMemers(generator, serializer);
    }

    @Override
    public <T> Set<T> getAllMembers(KeyGenerator generator, TypeReference<T> typeReference) {
        RedisSerializer<T> serializer = getSerializer(typeReference);
        return innerGetAllMemers(generator, serializer);
    }

    private <T> Set<T> innerGetAllMemers(KeyGenerator generator, RedisSerializer<T> redisSerializer) {
        Cursor cursor = operations.scan(generator.gen(), ScanOptions.scanOptions().count(10000).build());

        Set<T> result = new HashSet<>();
        while (cursor.hasNext()) {
            result.add(redisSerializer.deserialize((byte[]) cursor.next()));
        }

        return result;
    }

    private <T> Set<T> deserialize(Set<Object> data, Class<T> clasz) {
        Set<T> result = new HashSet<>();

        if (null != data && data.size() > 0) {
            data.forEach(item -> result.add((T) getSerializer(clasz).deserialize((byte[]) item)));
        }

        return result;
    }

    @Override
    public <T> Set<T> difference(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.difference(generator.gen(), convertKeys(otherGenerators)), clasz);
    }

    @Override
    public Long differenceAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.differenceAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public <T> Set<T> inter(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.intersect(generator.gen(), convertKeys(otherGenerators)), clasz);
    }

    @Override
    public Long interAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.intersectAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public <T> Set<T> union(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        return (Set<T>) deserialize(operations.union(generator.gen(), convertKeys(otherGenerators)), clasz);
    }

    @Override
    public Long unionAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        return operations.unionAndStore(generator.gen(), convertKeys(otherGenerators), storeGenerator.gen());
    }

    @Override
    public <T> Boolean move(KeyGenerator sourceKey, KeyGenerator targetKey, T item, Class<T> clasz) {
        return operations.move(sourceKey.gen(), Objects.requireNonNull(getSerializer(clasz).serialize(item)), targetKey.gen());
    }

    @Override
    public SetOperations getOriginal() {
        return operations;
    }

}
