package com.bxm.newidea.component.redis;

import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * 基于redis实现的分布式锁
 * @author liujia 2018/8/13 21:04
 */
@Component
public class DistributedLock {

    @Resource
    private RedisTemplate redisTemplate;

    @Autowired
    public DistributedLock(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    /**
     * 加锁，默认锁定5S
     * @param resource  锁定的资源
     * @param reqeustId 加锁的客户端ID，需保证唯一
     * @return
     */
    public boolean lock(String resource, String reqeustId) {
        return this.lock(resource, reqeustId, 5, TimeUnit.SECONDS);
    }

    /**
     * 加锁，自定义锁定时间
     * @param resource  锁定的资源
     * @param reqeustId 加锁的客户端ID，保证唯一
     * @param time      加锁的时长
     * @param timeUnit  加锁时间的单位
     * @return
     */
    public boolean lock(String resource, String reqeustId, long time, TimeUnit timeUnit) {
        String key = buildKey(resource);
        Boolean result = this.redisTemplate.opsForValue().setIfAbsent(key, reqeustId.getBytes());
        this.redisTemplate.expire(key, time, timeUnit);
        return result;
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    /**
     * 解锁
     * @param resource  解锁的资源
     * @param requestId 唯一码，用于确定是否为解锁的客户端
     * @return
     */
    public boolean unlock(String resource, String requestId) {
        String key = buildKey(resource);

        String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";

        byte[] requestByteArray = requestId.getBytes();
        Object result = this.redisTemplate.execute(new DefaultRedisScript(script, Long.class), Lists.newArrayList(key), requestByteArray);
        return new Long(1L).equals(result);
    }

}
