package com.bxm.newidea.component.redisson;

import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * 基于redssion实现的分布式锁
 *
 * @author liujia 2018/8/13 21:04
 */
@Component
@Slf4j
public class DistributedLock {

    @Resource
    private RedissonClient redissonClient;

    @Autowired
    public DistributedLock(RedissonClient redissonClient) {
        this.redissonClient = redissonClient;
    }

    /**
     * 锁定业务资源，防止并发情况对锁定的资源进行重复操作
     *
     * @param resource 锁定的资源
     * @return true表示加锁成功，可进行后续业务，false表示加锁失败，需要进行适当的重试机制
     */
    public Boolean lock(String resource) {
        return this.lock(resource, 5, TimeUnit.SECONDS);
    }


    /**
     * 锁定业务资源，防止并发情况对锁定的资源进行重复操作
     *
     * @param resource 锁定的资源,如业务ID、编码等
     * @param time     加锁的时长
     * @param timeUnit 加锁时间的单位
     * @return true表示加锁成功，可进行后续业务
     */
    public Boolean lock(String resource, long time, TimeUnit timeUnit) {
        String key = buildKey(resource);
        RLock rlock = redissonClient.getLock(key);
        if (rlock.isLocked()) {
            return false;
        }
        rlock.lock(time, timeUnit);
        return true;
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    /**
     * 尝试获取锁
     *
     * @param lockKey
     * @param unit
     * @param waitTime
     * @param leaseTime
     * @return
     */
    public boolean tryLock(String lockKey, TimeUnit unit, int waitTime, int leaseTime) {
        String key = buildKey(lockKey);
        RLock lock = redissonClient.getLock(key);
        try {
            return lock.tryLock(waitTime, leaseTime, unit);
        } catch (InterruptedException e) {
            return false;
        }
    }

    /**
     * 解锁资源
     *
     * @param resource 解锁的资源,如业务ID、编码等
     */
    public void unlock(String resource) {
        String key = buildKey(resource);
        RLock lock = redissonClient.getLock(key);
        try {
            lock.unlock();
        } catch (Exception e) {
            log.warn("当前kay锁资源不存在，解锁也没用，resource：{}", resource, e);
        }
    }

}
