package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.redisson.api.RSet;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

@Component
public class RedissonSetAdapterImpl extends BaseRedisAdapter implements RedisSetAdapter {


    @Autowired
    public RedissonSetAdapterImpl(RedissonClient redissonClient) {
        super(redissonClient);
    }

    private RSet setOperations(KeyGenerator generator) {
        return redissonClient.getSet(generator.gen());
    }

    @Override
    public Long add(KeyGenerator generator, Object... values) {
        AtomicLong deleteNum = new AtomicLong();
        for (Object value : values) {
            boolean add = setOperations(generator).add(this.serializerString(value));
            deleteNum.addAndGet(add ? 1 : 0);
        }
        return deleteNum.get();
    }

    @Override
    public <T> Long remove(KeyGenerator generator, Object... values) {
        AtomicLong deleteNum = new AtomicLong();
        for (Object value : values) {
            boolean add = setOperations(generator).remove(this.serializerString(value));
            deleteNum.addAndGet(add ? 1 : 0);
        }
        return deleteNum.get();
    }

    @Override
    public <T> Boolean exists(KeyGenerator generator, T value) {
        return setOperations(generator).contains(this.serializerString(value));
    }

    @Override
    public <T> List<T> pop(KeyGenerator generator, Long count, Class<T> clasz) {
        Set<String> result = setOperations(generator).removeRandom(count.intValue());
        List<T> finalResult = Lists.newArrayList();
        if (result != null) {
            finalResult = result.stream().map(value -> this.deSerializerString(value, clasz, null)).collect(Collectors.toList());
        }
        return finalResult;
    }

    @Override
    public <T> List<T> pop(KeyGenerator generator, Long count, TypeReference<T> typeReference) {
        Set<String> result = setOperations(generator).removeRandom(count.intValue());
        List<T> finalResult = Lists.newArrayList();
        if (result != null) {
            finalResult = result.stream().map(value -> this.deSerializerString(value, null, typeReference)).collect(Collectors.toList());
        }
        return finalResult;
    }


    @Override
    public Long size(KeyGenerator generator) {
        return (long) setOperations(generator).size();
    }

    @Override
    public <T> Set<T> getAllMembers(KeyGenerator generator, Class<T> clasz) {
        return innerGetAllMemers(generator, clasz, null);
    }

    @Override
    public <T> Set<T> getAllMembers(KeyGenerator generator, TypeReference<T> typeReference) {
        return innerGetAllMemers(generator, null, typeReference);
    }

    private <T> Set<T> innerGetAllMemers(KeyGenerator generator, Class<T> clasz, TypeReference<T> typeReference) {
        Set<String> set = setOperations(generator).readAll();
        Set<T> result = new HashSet<>();
        Iterator<String> iterator = set.iterator();
        while (iterator.hasNext()) {
            result.add(this.deSerializerString(iterator.next(), clasz, typeReference));
        }
        return result;
    }


    @Override
    public <T> Set<T> difference(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        Set<String> set = setOperations(generator).readDiff(convertKeys(otherGenerators).stream().toArray(String[]::new));
        Set<T> result = new HashSet<>();
        for (String s : set) {
            result.add(this.deSerializerString(s, clasz, null));
        }
        return result;
    }

    @Override
    public Long differenceAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }
        Set<String> set = setOperations(generator).readDiff(convertKeys(otherGenerators).stream().toArray(String[]::new));
        AtomicLong deleteNum = new AtomicLong();
        for (String value : set) {
            boolean add = setOperations(storeGenerator).add(value);
            deleteNum.addAndGet(add ? 1 : 0);
        }
        return deleteNum.get();
    }

    @Override
    public <T> Set<T> inter(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        Set<String> set = setOperations(generator).readIntersection(convertKeys(otherGenerators).stream().toArray(String[]::new));
        Set<T> result = new HashSet<>();
        for (String s : set) {
            result.add(this.deSerializerString(s, clasz, null));
        }
        return result;
    }

    @Override
    public Long interAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }

        Set<String> set = setOperations(generator).readIntersection(convertKeys(otherGenerators).stream().toArray(String[]::new));
        AtomicLong deleteNum = new AtomicLong();
        for (String value : set) {
            boolean add = setOperations(storeGenerator).add(value);
            deleteNum.addAndGet(add ? 1 : 0);
        }
        return deleteNum.get();
    }

    @Override
    public <T> Set<T> union(KeyGenerator generator, Class<T> clasz, KeyGenerator... otherGenerators) {
        Set<String> set = setOperations(generator).readUnion(convertKeys(otherGenerators).stream().toArray(String[]::new));
        Set<T> result = new HashSet<>();
        for (String s : set) {
            result.add(this.deSerializerString(s, clasz, null));
        }
        return result;
    }

    @Override
    public Long unionAndStore(KeyGenerator storeGenerator, KeyGenerator generator, KeyGenerator... otherGenerators) {
        if (storeGenerator == null) {
            return 0L;
        }
        Set<String> set = setOperations(generator).readUnion(convertKeys(otherGenerators).stream().toArray(String[]::new));
        AtomicLong deleteNum = new AtomicLong();
        for (String value : set) {
            boolean add = setOperations(storeGenerator).add(value);
            deleteNum.addAndGet(add ? 1 : 0);
        }
        return deleteNum.get();
    }

    @Override
    public <T> Boolean move(KeyGenerator sourceKey, KeyGenerator targetKey, T item, Class<T> clasz) {
        return setOperations(sourceKey).move(targetKey.gen(), this.serializerString(item));
    }


}
