package com.bxm.newidea.component.redisson.config;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.cglib.proxy.Enhancer;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * 创建RedissonClientBeanDefinition实例工厂
 *
 * @author wzy
 * @date 2021/8/5 2:46 下午
 **/
@Slf4j
public class RedissonClientBeanDefinitionFactory {

    /**
     * 持有RedissonClient集合的类
     */
    private RedissonClientHolder redissonClientHolder;

    public RedissonClientBeanDefinitionFactory(RedissonClientHolder redissonClientHolder) {
        this.redissonClientHolder = redissonClientHolder;
    }

    /**
     * 获取对应数据源的RedissonClient的BeanDefinition
     *
     * @param dataSourceName   数据源名称
     * @param redissonMetaInfo 当前数据源配置源信息
     * @return Bean定义
     */
    public BeanDefinition get(String dataSourceName, RedissonMetaInfo redissonMetaInfo, String[] activeProfiles) {
        //创建beanDefinition
        AbstractBeanDefinition redissonClientBeanDefinition = BeanDefinitionBuilder.genericBeanDefinition(RedissonClient.class,
                () -> getRedissonClientBean(dataSourceName, redissonMetaInfo, activeProfiles)).getBeanDefinition();

        Objects.requireNonNull(redissonClientBeanDefinition,
                String.format("redisson对应的BeanDefinition不存在,数据源名称：%s", dataSourceName));

        return redissonClientBeanDefinition;
    }

    /**
     * 延迟初始化bean，使用aop方式去创建代理动态代理对象
     *
     * @param dataSourceName   数据源名称
     * @param redissonMetaInfo 配置源信息
     * @param activeProfiles   激活的profiles
     * @return RedissonClient对象
     */
    private RedissonClient getRedissonClientBean(String dataSourceName, RedissonMetaInfo redissonMetaInfo, String[] activeProfiles) {
        List<String> targetProfiles = Stream.of(activeProfiles).filter(activeProfile ->
                StringUtils.equalsAnyIgnoreCase(activeProfile, "LOCAL", "DEV"))
                .collect(Collectors.toList());
        // 本地环境使用延迟加载
        if (!CollectionUtils.isEmpty(targetProfiles)) {
            Enhancer enhancer = new Enhancer();
            enhancer.setSuperclass(RedissonClient.class);
            enhancer.setCallback(new RedissonMethodInterceptor(redissonClientHolder, redissonMetaInfo, dataSourceName));

            return (RedissonClient) enhancer.create();
        }

        return redissonClientHolder.get(dataSourceName, redissonMetaInfo);

    }
}
