package com.bxm.newidea.component.redisson.config;

import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RedissonClient;
import org.springframework.context.ApplicationContext;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/8/23 7:09 下午
 */
@Slf4j
public class SwitchMultiDataSourceHandler {

    private final ApplicationContext applicationContext;

    private static Map<String, RedissonClient> cacheRedissonClientMap = new ConcurrentHashMap<>();

    public SwitchMultiDataSourceHandler(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    /**
     * 获取Redisson客户端对象
     *
     * @return Redisson客户端对象
     */
    public RedissonClient getRedissonClient(String dbName) {
        //先从缓存map里面取指定数据源client
        RedissonClient targetRedissonClient = cacheRedissonClientMap.get(dbName);

        if (Objects.nonNull(targetRedissonClient)) {
            return targetRedissonClient;
        }

        //再从spring里面取指定数据源client
        targetRedissonClient = applicationContext.getBean(dbName + "RedissonDataSource", RedissonClient.class);
        if (Objects.nonNull(targetRedissonClient)) {
            cacheRedissonClientMap.put(dbName, targetRedissonClient);
            return targetRedissonClient;
        }

        //再从缓存中取默认连接
        RedissonClient defaultRedissonClient = cacheRedissonClientMap.get(dbName);
        //使用默认的
        if (Objects.nonNull(defaultRedissonClient)) {
            return defaultRedissonClient;
        }

        //再从Spring中取默认连接
        targetRedissonClient = applicationContext.getBean("default" + "RedissonDataSource", RedissonClient.class);
        if (Objects.nonNull(targetRedissonClient)) {
            cacheRedissonClientMap.put("default", targetRedissonClient);
            return targetRedissonClient;
        }

        //走到这一步不可能发生，打印错误日志
        log.error("未设置Redisson默认数据源，请检查配置！");

        return targetRedissonClient;
    }
}