package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.DistributedLock;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * 基于redis实现的分布式锁
 * 与线程绑定，根据线程信息作为加锁、解锁依据
 *
 * @author liujia 2018/8/13 21:04
 */
@Component
@Slf4j
public class DistributedLockImpl implements DistributedLock {

    @Resource
    private RedisTemplate redisTemplate;

    @Autowired
    public DistributedLockImpl(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    private String getRequestId() {
        return Thread.currentThread().getName() + ":" + Thread.currentThread().getId();
    }

    @Override
    public boolean lock(String resource) {
        return lock(resource, 5, TimeUnit.SECONDS);
    }

    @Override
    public boolean lock(String resource, long time, TimeUnit timeUnit) {

        String key = buildKey(resource);
        boolean result = this.redisTemplate.opsForValue().setIfAbsent(key, getRequestId().getBytes());
        this.redisTemplate.expire(key, time, timeUnit);
        return result;
    }

    @Override
    public void unlock(String resource) {
        String key = buildKey(resource);

        String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";

        byte[] requestByteArray = getRequestId().getBytes();
        Object result = this.redisTemplate.execute(new DefaultRedisScript(script, Long.class), Lists.newArrayList(key), requestByteArray);
        if (!new Long(1L).equals(result)) {
            if (log.isDebugEnabled()) {
                log.debug("[{}]解锁失败", resource);
            }
        }
    }
}
