package com.bxm.newidea.component.schedule.config;

import com.netflix.appinfo.EurekaInstanceConfig;
import com.xxl.job.core.executor.impl.XxlJobSpringExecutor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.cloud.commons.util.InetUtils;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * xxl-job config
 * @author xuxueli 2017-04-28
 */
@Data
@Slf4j
@Configuration
@ConfigurationProperties(prefix = "xxl.job.executor")
public class XxlJobConfiguration {
    private static final String EXECUTOR_IP = "executorIp";

    private static final int DEFAULT_PORT = 19099;

    /**
     * 调度服务地址，多个地址使用英文逗号分隔
     * 必须配置
     */
    private String adminAddresses;

    /**
     * 当前应用的名称
     * 如未配置则将获取当前服务的默认服务名
     */
    private String appName;

    /**
     * 引用该模块的服务部署设备实际IP，用于调度服务回调
     * 如未配置将尝试通过网卡获取
     */
    private String ip;

    /**
     * 对调度服务暴露的端口，用于调度服务回调
     * 如未配置将使用默认端口，注意多服务部署时的端口冲突
     */
    private int port = DEFAULT_PORT;

    /**
     * 访问凭证，如果配置需要与调度服务保持一致，用于安全性数据传输
     * 非必须，可以为空
     */
    private String accessToken;

    /**
     * 日志文件路径
     */
    private String logPath;

    /**
     * 日志清理周期
     */
    private int logRetentionDays;

    private final EurekaInstanceConfig instance;

    private final InetUtils inetUtils;

    @Autowired(required = false)
    public XxlJobConfiguration(EurekaInstanceConfig instance, InetUtils inetUtils) {
        this.instance = instance;
        this.inetUtils = inetUtils;
    }

    @Bean(initMethod = "start", destroyMethod = "destroy")
    public XxlJobSpringExecutor xxlJobExecutor() {
        XxlJobSpringExecutor xxlJobSpringExecutor = new XxlJobSpringExecutor();

        if (StringUtils.isBlank(appName)) {
            if (null != instance) {
                appName = instance.getAppname();
            } else {
                throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.app-name配置属性");
            }
        }

        if (StringUtils.isBlank(ip)) {
            ip = System.getProperty(EXECUTOR_IP);
            if (StringUtils.isBlank(ip)) {
                if (null != inetUtils) {
                    ip = inetUtils.findFirstNonLoopbackHostInfo().getIpAddress();
                } else {
                    throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.ip配置属性");
                }
            }
        }

        if (StringUtils.isBlank(adminAddresses)) {
            throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.admin-address配置属性");
        }

        if (DEFAULT_PORT == port) {
            log.info("调度服务回调接口使用了默认的[{}]端口，请注意部署时的端口冲突问题", DEFAULT_PORT);
        }

        xxlJobSpringExecutor.setAdminAddresses(adminAddresses);
        xxlJobSpringExecutor.setAppName(appName);
        xxlJobSpringExecutor.setIp(ip);
        xxlJobSpringExecutor.setPort(port);
        xxlJobSpringExecutor.setAccessToken(accessToken);
        xxlJobSpringExecutor.setLogPath(logPath);
        xxlJobSpringExecutor.setLogRetentionDays(logRetentionDays);

        return xxlJobSpringExecutor;
    }
}