package com.bxm.component.spring;

import com.bxm.component.spring.event.ShutdownEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;

/**
 * 魔改 spring application shutdown hook
 * @author Gonzo
 * @date 2019-12-04 15:46
 */
@Slf4j
public class ShutdownHook extends Thread {

    private ApplicationContext applicationContext;

    public ShutdownHook(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
        // 注册shutdown hook
        registerShutdownHook();
    }

    /**
     * 将默认的application context的shutdown hook设置为自定义的
     */
    private void registerShutdownHook() {
        log.info("注册自定义 shutdown hook");
        try {
            Field shutdownHook = ReflectionUtils.findField(AbstractApplicationContext.class, "shutdownHook");
            ReflectionUtils.makeAccessible(shutdownHook);
            shutdownHook.set(this.applicationContext, this);

        } catch (IllegalAccessException e) {
            log.error("自定义 shutdownHook 失败", e);
        }
        Runtime.getRuntime().addShutdownHook(this);
    }

    /**
     * 自定义的shutdown实现
     * 在原本的shutdown前，调用自定义的shutdown
     */
    @Override
    public void run() {
        log.info("运行自定义 shutdown hook");
        // 在Spring close之前
        applicationContext.publishEvent(new ShutdownEvent(applicationContext));

        log.info("运行spring shutdown hook");
        // 最终调用spring的shutdown
        Assert.isInstanceOf(ConfigurableApplicationContext.class, applicationContext);
        ((ConfigurableApplicationContext) applicationContext).close();
    }
}
