package com.bxm;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.enums.FileTypeEnum;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.FileUtils;
import com.bxm.newidea.component.vo.ResponseJson;
import com.fasterxml.jackson.core.type.TypeReference;
import org.junit.Assert;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.ResultActions;
import org.springframework.test.web.servlet.request.MockHttpServletRequestBuilder;
import org.springframework.test.web.servlet.request.MockMultipartHttpServletRequestBuilder;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.ServletContext;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.Date;

import static org.apache.commons.io.FileUtils.openInputStream;

/**
 * 测试controller的基类
 *
 * @author liujia 2018/8/1 21:10
 */
@RunWith(SpringRunner.class)
@SpringBootTest
@ActiveProfiles({"LOCAL", "TEST"})
@AutoConfigureMockMvc
public class BaseTestSupporter {

    @Autowired
    public WebApplicationContext webApplicationContext;

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    protected MockMvc mockMvc;


    private ServletContext servletContext;

    @Before
    public void setUp() {
        this.servletContext = this.webApplicationContext.getServletContext();
    }

    protected <Z> Z getJson(MvcResult result, Class<Z> clasz) {
        try {
            return JSON.parseObject(result.getResponse().getContentAsString(), clasz);
        } catch (UnsupportedEncodingException e) {
            this.logger.error(e.getMessage(), e);
            return null;
        }
    }

    protected <Z> Z getJson(MockHttpServletRequestBuilder requestBuilder, Class<Z> clasz) {
        return getJson(perform(requestBuilder).andReturn(), clasz);
    }

    protected ResponseJson getJson(MockHttpServletRequestBuilder requestBuilder) {
        return getJson(perform(requestBuilder).andReturn(), ResponseJson.class);
    }

    protected ResultActions perform(MockHttpServletRequestBuilder request) {
        try {
            return this.mockMvc.perform(request);
        } catch (Exception e) {
            this.logger.error(e.getMessage(), e);
        }
        return null;
    }

    protected <Z> ResponseJson<Z> getJson(MockHttpServletRequestBuilder requestBuilder, TypeReference<ResponseJson<Z>> typeReference) {
        return getJson(perform(requestBuilder).andReturn(), typeReference);
    }

    protected <Z> ResponseJson<Z> getJson(MvcResult result, TypeReference<ResponseJson<Z>> typeReference) {
        try {
            return JSON.parseObject(result.getResponse().getContentAsString(), typeReference);
        } catch (UnsupportedEncodingException e) {
            this.logger.error(e.getMessage(), e);
            return null;
        }

    }

    protected <T> void assertResponseOk(ResponseJson<T> result) {
        Assert.assertTrue(result.getCode() == HttpStatus.OK.value());
    }

    /**
     * 将对象的值填充到请求构造器中
     *
     * @param request  请求信息
     * @param paramObj 填充的对象，暂未处理二级对象与对象嵌套
     */
    protected <T> void buildRequest(MockHttpServletRequestBuilder request, T paramObj) {
        deepObjBuild(request, null, 0, paramObj);
    }

    protected void buildPostReqeust(MockHttpServletRequestBuilder request, Object param) {
        request.contentType(MediaType.APPLICATION_JSON_UTF8);
        request.content(JSON.toJSONString(param));
    }

    private void buildArray(MockHttpServletRequestBuilder request, String fieldName, Object array) {
        if (Collection.class.isAssignableFrom(array.getClass())) {
            Collection collectionObj = (Collection) array;
            int index = 0;
            for (Object element : collectionObj) {
                deepObjBuild(request, fieldName, index, element);
                index++;
            }
        }
    }

    private void deepObjBuild(MockHttpServletRequestBuilder request, String parentFieldName, int index, Object childObj) {
        Method[] allDeclaredMethods = ReflectionUtils.getAllDeclaredMethods(childObj.getClass());
        for (Method declaredMethod : allDeclaredMethods) {
            Class<?> returnType = declaredMethod.getReturnType();
            if (!Void.TYPE.isAssignableFrom(returnType)) {
                String fieldName = com.bxm.newidea.component.tools.ReflectionUtils.getFieldName(declaredMethod.getName());

                if (fieldName == null) {
                    continue;
                }

                Object value = ReflectionUtils.invokeMethod(declaredMethod, childObj);

                if (null == value) {
                    continue;
                }

                if (null != parentFieldName) {
                    fieldName = parentFieldName + "[" + index + "]." + fieldName;
                }

                if (MultipartFile.class.isAssignableFrom(returnType)) {
                    if (request instanceof MockMultipartHttpServletRequestBuilder) {
                        ((MockMultipartHttpServletRequestBuilder) request).file((MockMultipartFile) value);
                    } else {
                        this.logger
                                .error("设置的对象包含文件，但是请求不是MockMultipartHttpServletRequestBuilder的实例，使用MockMvcRequestBuilders.multipart构建符合条件的实例");
                    }
                } else if (Date.class.isAssignableFrom(returnType)) {
                    request.param(fieldName, DateUtils.formatDateTime((Date) value));
                } else if (BigDecimal.class.isAssignableFrom(returnType)) {
                    request.param(fieldName, value.toString());
                } else if (com.bxm.newidea.component.tools.ReflectionUtils.isPrimitiveType(returnType)) {
                    request.param(fieldName, value.toString());
                } else if (Collection.class.isAssignableFrom(returnType)) {
                    buildArray(request, fieldName, value);
                } else if (!Class.class.isAssignableFrom(returnType)) {
                    this.logger.info("忽略属性值[{}],其对象类型是[{}]", fieldName, returnType);
                }
            }
        }
    }

    /**
     * 根据test-classes作为根目录，读取filePath对应的文件
     *
     * @param filePath  resource下读取的文件路径，包括文件名
     * @param fieldName 接收的字段名称
     * @return 模拟文件
     */
    protected MockMultipartFile createMockFile(String filePath, String fieldName) {
        String finalFilePath = FileUtils.getClassRootPath() + File.separator + filePath;

        String fileName = FileUtils.getFileName(filePath);
        MockMultipartFile mockFile = null;
        try {
            String contextType = FileTypeEnum.getContentType(FileUtils.getFilextension(fileName));
            mockFile = new MockMultipartFile(fieldName,
                    fileName,
                    contextType,
                    openInputStream(new File(finalFilePath)));
        } catch (IOException e) {
            this.logger.error("文件不存在:" + finalFilePath, e);
        }
        return mockFile;
    }
}
