package com.bxm.newidea.component.filter;

import com.bxm.newidea.component.annotations.FilterBean;
import com.bxm.newidea.component.exception.ExcutorException;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.OrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 过滤链执行，会完整的执行过滤链
 *
 * @author liujia
 * @date 1/13/21 6:23 PM
 **/
@Component
@Slf4j
public class FilterChainExecutor implements ApplicationRunner {

    private Map<String, List<IFilter>> groupFilterMap = new HashMap<>();

    /**
     * 执行指定业务分组的过滤器
     *
     * @param group   业务分组
     * @param context 过滤器执行上下文
     */
    @SuppressWarnings("unchecked")
    public <T> void doFilter(String group, T context) {
        List<IFilter> filters = groupFilterMap.get(group);

        if (null == filters) {
            return;
        }

        for (IFilter filter : filters) {
            try {
                filter.doFilter(context);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new ExcutorException("过滤器执行失败：" + filter.getClass().getSimpleName(), e);
            }
        }
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        for (IFilter filter : SpringContextHolder.getBeans(IFilter.class)) {
            FilterBean annotation = AnnotationUtils.findAnnotation(filter.getClass(), FilterBean.class);
            if (null == annotation) {
                log.warn("[{}]未提供FilterBean注解", filter.getClass().getSimpleName());
                continue;
            }

            List<IFilter> filters = groupFilterMap.get(annotation.group());
            if (filters == null) {
                filters = new ArrayList<>();
            }
            filters.add(filter);
            groupFilterMap.put(annotation.group(), filters);
        }

        for (List<IFilter> rules : groupFilterMap.values()) {
            OrderComparator.sort(rules);
        }
    }
}
