package com.bxm.newidea.component.rule;

import com.bxm.newidea.component.annotations.RuleBean;
import com.bxm.newidea.component.exception.ExcutorException;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.core.OrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 规则分组执行器
 *
 * @author liujia
 * @date 1/13/21 6:03 PM
 **/
@Component
@Slf4j
public class RuleGroupExecutor implements ApplicationRunner {

    private ApplicationContext applicationContext;

    private Map<String, List<IRule>> groupRuleMap = new HashMap<>();

    /**
     * 执行对应业务分组的规则
     *
     * @param groupName 业务分组名称，与{@link IRule}实现类上的注解保持一致
     * @param context   规则执行的上下文
     * @return 规则匹配结果
     */
    @SuppressWarnings("unchecked")
    public <T> boolean apply(String groupName, T context) {
        List<IRule> rules = groupRuleMap.get(groupName);

        if (null == rules) {
            log.error("不存在[{}]对应的逻辑规则分组", groupName);
            return false;
        }

        for (IRule rule : rules) {
            try {
                if (!rule.apply(context)) {
                    return false;
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new ExcutorException("业务逻辑规则执行失败，规则类：" + rule.getClass().getSimpleName(), e);
            }
        }
        return true;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        for (IRule rule : SpringContextHolder.getBeans(IRule.class)) {
            RuleBean annotation = AnnotationUtils.findAnnotation(rule.getClass(), RuleBean.class);
            if (null == annotation) {
                log.warn("[{}]未提供RuleBean注解", rule.getClass().getSimpleName());
                continue;
            }

            List<IRule> rules = groupRuleMap.get(annotation.group());
            if (rules == null) {
                rules = new ArrayList<>();
            }
            rules.add(rule);
            groupRuleMap.put(annotation.group(), rules);
        }

        for (List<IRule> rules : groupRuleMap.values()) {
            OrderComparator.sort(rules);
        }
    }
}
