package com.bxm.component.tbk.order.api;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.tbk.order.config.TbkOpenOrderProperties;
import com.bxm.component.tbk.order.model.enums.OrderTypeEnum;
import com.bxm.component.tbk.order.model.resp.RefundResponse;
import com.bxm.component.tbk.order.model.dto.TbkOrderInfo;
import com.bxm.component.tbk.order.model.dto.TbkOrderPageInfo;
import com.bxm.newidea.component.tools.DateUtils;
import com.taobao.api.ApiException;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import com.taobao.api.request.TbkOrderDetailsGetRequest;
import com.taobao.api.request.TbkRelationRefundRequest;
import com.taobao.api.response.TbkOrderDetailsGetResponse;
import com.taobao.api.response.TbkRelationRefundResponse;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;

@Slf4j
@Service
@AllArgsConstructor
public class TbkOrderApi {

    private final TbkOpenOrderProperties tbkOpenOrderProperties;

    private final TaobaoClient client;

    public TbkOrderPageInfo<TbkOrderInfo> pullOrderNew(Long pageNo, Long pageSize, String positionIndex,
                                                       Date orderStartTime, Date orderEndTime, int type) {
        return pullOrderNew(pageNo, pageSize, positionIndex, DateUtils.formatDateTime(orderStartTime),
                DateUtils.formatDateTime(orderEndTime), type);
    }

    public TbkOrderPageInfo<TbkOrderInfo> pullOrderNew(Long pageNo, Long pageSize, String positionIndex,
                                                       LocalDateTime orderStartTime, LocalDateTime orderEndTime,
                                                       int type) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);

        return pullOrderNew(pageNo, pageSize, positionIndex, formatter.format(orderStartTime),
                formatter.format(orderEndTime), type);
    }

    public TbkOrderPageInfo<TbkOrderInfo> pullOrderNew(Long pageNo, Long pageSize, String positionIndex,
                                                       String orderStartTime, String orderEndTime, int type) {

        TbkOrderPageInfo<TbkOrderInfo> tbOrderPageInfo = new TbkOrderPageInfo<>();

        TbkOrderDetailsGetRequest req = new TbkOrderDetailsGetRequest();
        req.setQueryType(1L);
        //下一页必须传入positionIndex
        req.setPositionIndex(positionIndex);
        req.setPageSize(pageSize);
        req.setPageNo(pageNo);
        req.setStartTime(orderStartTime);
        req.setEndTime(orderEndTime);
        //tk_status不传表示查询所有状态
        //推广者角色类型-2:二方，3:三方，不传，表示所有角色
        req.setMemberType(2L);
        //渠道订单
        convertSceneType(req, type);

        TbkOrderDetailsGetResponse rsp;
        try {
            rsp = client.execute(req);
        } catch (ApiException e) {
            log.error("拉取订单异常, 拉取时间:{}", orderStartTime, e);
            return null;
        }

        if (null == rsp || !rsp.isSuccess() || CollectionUtils.isEmpty(rsp.getData().getResults())) {
            log.info("从淘宝获取获取订单为空或者失败,查询时间为:{},错误信息:{}", orderStartTime, JSON.toJSON(rsp));
            return null;
        }
        List<TbkOrderDetailsGetResponse.PublisherOrderDto> resultRsp = rsp.getData().getResults();
        List<TbkOrderInfo> orderInfoList = new ArrayList<>();
        for (TbkOrderDetailsGetResponse.PublisherOrderDto publisherOrderDto : resultRsp) {
            orderInfoList.add(convertOrderInfoNew(publisherOrderDto, type));
        }

        TbkOrderDetailsGetResponse.OrderPage tbOrderPage = rsp.getData();
        tbOrderPageInfo.setHasNext(tbOrderPage.getHasNext());
        tbOrderPageInfo.setHasPre(tbOrderPage.getHasPre());
        tbOrderPageInfo.setPageNo(tbOrderPage.getPageNo());
        tbOrderPageInfo.setPageSize(tbOrderPage.getPageSize());
        tbOrderPageInfo.setPositionIndex(tbOrderPage.getPositionIndex());
        tbOrderPageInfo.setList(orderInfoList);
        return tbOrderPageInfo;
    }


    public List<RefundResponse> pullRefundOrder(Long pageSize, Long searchType, Long refundType,
                                                Long pageNo, Long bizType, Date startTime) {
        TaobaoClient client = new DefaultTaobaoClient(tbkOpenOrderProperties.getTbOpenOrderUrl(),
                tbkOpenOrderProperties.getAppKey(), tbkOpenOrderProperties.getAppSecret());
        TbkRelationRefundRequest req = new TbkRelationRefundRequest();
        TbkRelationRefundRequest.TopApiRefundRptOption obj1 = new TbkRelationRefundRequest.TopApiRefundRptOption();
        obj1.setPageSize(pageSize);
        obj1.setSearchType(searchType);
        obj1.setRefundType(refundType);
        obj1.setStartTime(startTime);
        obj1.setPageNo(pageNo);
        obj1.setBizType(bizType);
        req.setSearchOption(obj1);

        log.info("查询退款信息 请求参数: {}", JSON.toJSONString(req));

        try {
            TbkRelationRefundResponse rsp = client.execute(req);

            if (log.isDebugEnabled()) {
                log.debug("查询退款信息返回: {}", JSON.toJSONString(rsp));
            }

            while (true) {
                if (Objects.nonNull(rsp) && rsp.isSuccess() && StringUtils.isNotBlank(rsp.getBody())) {

                    JSONObject jsonObject = JSON.parseObject(rsp.getBody());
                    if (Objects.nonNull(jsonObject)) {
                        // tbk_relation_refund_response.result.data.results.result 才是真实数据
                        String tbkRelationRefundResponse = jsonObject.getString("tbk_relation_refund_response");

                        if (StringUtils.isNotBlank(tbkRelationRefundResponse)
                                && Objects.nonNull(jsonObject = JSON.parseObject(tbkRelationRefundResponse))) {
                            String result = jsonObject.getString("result");

                            if (StringUtils.isNotBlank(tbkRelationRefundResponse)
                                    && Objects.nonNull(jsonObject = JSON.parseObject(result))) {

                                String data = jsonObject.getString("data");
                                if (StringUtils.isNotBlank(data)
                                        && Objects.nonNull(jsonObject = JSON.parseObject(data))) {
                                    String results = jsonObject.getString("results");
                                    if (StringUtils.isNotBlank(data)
                                            && Objects.nonNull(jsonObject = JSON.parseObject(results))) {
                                        String finalResult = jsonObject.getString("result");

                                        if (StringUtils.isNotBlank(finalResult)) {
                                            return JSON.parseArray(finalResult, RefundResponse.class);
                                        }

                                    }
                                }
                            }
                        }
                    }
                    log.warn("查询退款信息失败 请求参数: {} 返回参数: {}", JSON.toJSONString(req), JSON.toJSONString(rsp));
                    return Collections.emptyList();
                }
            }

        } catch(Exception e) {
            log.error("查询退款信息失败， 请求参数: {}", JSON.toJSONString(req), e);
        }

        return Collections.emptyList();
    }

    public List<RefundResponse> pullRefundOrder(Long pageSize, Long searchType, Long refundType,
                                                                  Long pageNo, Long bizType,
                                                                  LocalDateTime startTime) {
        return pullRefundOrder(pageSize, searchType, refundType, pageNo, bizType,
                Date.from(startTime.atZone(ZoneId.systemDefault()).toInstant()));
    }

    /**
     * 转换淘宝订单数据到库中订单数据结果集
     * @param tbOrderInfo 淘宝订单信息
     * @param type 订单类型
     * @return
     */
    private TbkOrderInfo convertOrderInfoNew(TbkOrderDetailsGetResponse.PublisherOrderDto tbOrderInfo,int type){
        StringBuilder httpsPre = new StringBuilder("https:");

        TbkOrderInfo orderInfo = TbkOrderInfo.builder()
                .goodsId(Objects.toString(tbOrderInfo.getItemId(), null))
                .goodsName(tbOrderInfo.getItemTitle())
                .relationId(Objects.toString(tbOrderInfo.getRelationId(), null))
                .specialId(Objects.toString(tbOrderInfo.getSpecialId(), null))
                .orderSn(Objects.toString(tbOrderInfo.getTradeId(), null))
                .orderParentSn(Objects.toString(tbOrderInfo.getTradeParentId(), null))
                .goodsPrice(StringUtils.isBlank(tbOrderInfo.getItemPrice()) ? null : new BigDecimal(tbOrderInfo.getItemPrice()))
                // 商品购买数量
                .goodsNum(tbOrderInfo.getItemNum())
                //实际支付价格
                .payPrice(StringUtils.isBlank(tbOrderInfo.getAlipayTotalPrice()) ? new BigDecimal("0.00")
                        : new BigDecimal(tbOrderInfo.getAlipayTotalPrice()))
                //时间转换
                .sourceOrderCreateTime(DateUtils.parseDateTime(tbOrderInfo.getTkCreateTime()))
                .sourceOrderEarningTime(DateUtils.parseDateTime(tbOrderInfo.getTkEarningTime()))
                .commission(StringUtils.isBlank(tbOrderInfo.getPubSharePreFee()) ? null : new BigDecimal(tbOrderInfo.getPubSharePreFee()))
                .sourceOwnerOrderStatus(null == tbOrderInfo.getTkStatus() ? null : tbOrderInfo.getTkStatus().intValue())
                .source("TB")
                //新接口提供了商品图片地址
                .imgUrl(httpsPre.append(tbOrderInfo.getItemImg()).toString())
                .pid(parsePid(tbOrderInfo))
                .adzoneId(Objects.toString(tbOrderInfo.getAdzoneId()))
                .siteId(Objects.toString(tbOrderInfo.getSiteId()))
                .type(type)
                .tbOrderType(tbOrderInfo.getOrderType())
                .createTime(new Date())
                .sourceStr(JSON.toJSONString(tbOrderInfo))
                .build();

        return orderInfo;
    }

    private String parsePid(TbkOrderDetailsGetResponse.PublisherOrderDto tbOrderInfo) {
        StringBuilder pid = new StringBuilder("mm_456760114_")
                .append(tbOrderInfo.getSiteId())
                .append("_")
                .append(tbOrderInfo.getAdzoneId());

        return pid.toString();
    }

    /**
     * 根据定义的类型设置不同的查询参数
     * @param req 淘宝查询参数
     * @param orderType 查询的订单类型
     */
    private void convertSceneType(TbkOrderDetailsGetRequest req, int orderType) {

        //推广者角色类型-为不传-全部角色 --branch/ticket中产品确认更改
        //orderScene 1:常规订单，2:渠道订单，3:会员运营订单，默认为1
        if(OrderTypeEnum.CHANNEL_ORDER.getCode() == orderType){
            //场景订单类型为2渠道订单
            req.setOrderScene(2L);
            return;
        }

        if(OrderTypeEnum.GENERAL_ORDER.getCode() == orderType){
            //场景订单类型为1常规订单
            req.setOrderScene(1L);
            return;
        }

        //默认类型 若上面的都不匹配 默认为渠道订单查询类型
        req.setOrderScene(2L);
    }
}
