package com.bxm.component.tbk.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.api.TbkOrderApi;
import com.bxm.component.tbk.order.config.OrderTimerConfigProperties;
import com.bxm.component.tbk.order.config.TbkOpenOrderProperties;
import com.bxm.component.tbk.order.model.dto.LongTimeNoCheckOrderInfo;
import com.bxm.component.tbk.order.model.dto.TbkOrderInfo;
import com.bxm.component.tbk.order.model.dto.TbkOrderPageInfo;
import com.bxm.component.tbk.order.model.enums.BusinesSiteId;
import com.bxm.component.tbk.order.model.enums.OrderTypeEnum;
import com.bxm.component.tbk.order.model.enums.TbkOrderType;
import com.bxm.component.tbk.order.service.EleTakeOutOrderService;
import com.bxm.component.tbk.order.service.PullOrderService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Objects;

/**
 * 商品、饿了么订单拉取
 * @author Gonzo
 * @date  2020-05-03 14:43
 * @since 2.0.0
 */
@Slf4j
public abstract class AbstractPullOrderService implements PullOrderService {

    @Autowired
    private TbkOrderApi tbkOrderApi;

    @Autowired
    private EleTakeOutOrderService eleTakeOutOrderService;

    @Autowired
    private OrderTimerConfigProperties orderTimerConfigProperties;

    @Autowired
    private TbkOpenOrderProperties tbkOpenOrderProperties;


    @Override
    public void pullOrder() {
        if (!orderTimerConfigProperties.getPullOrderSwitch()) {
            log.info("每2分钟订单拉取开关关闭,不进行拉取");
            return;
        }

        // 开始时间，提前15分钟
        queryApiAndUpdate(LocalDateTime.now().minusMinutes(15),
                // 结束时间 现在
                LocalDateTime.now(),
                // 查询渠道订单
                OrderTypeEnum.CHANNEL_ORDER);
    }

    @Override
    public void pullOrderByDay() {
        if (!orderTimerConfigProperties.getPullOrderDaySwitch()) {
            log.info("每半天订单拉取开关关闭,不进行拉取");
            return;
        }

        // 开始时间，提前8个小时
        queryApiAndUpdateByDays(LocalDateTime.now().minusHours(8),
                // 结束时间 现在 - 1h 避免和每2分钟处理的冲突
                LocalDateTime.now().minusHours(1),
                // 查询渠道订单
                OrderTypeEnum.CHANNEL_ORDER);
    }

    @Override
    public void pullGeneralOrder() {
        if (!orderTimerConfigProperties.getPullOrderSwitchGeneral()) {
            log.info("常规订单拉取开关关闭,不进行拉取");
            return;
        }

        // 开始时间，提前三分钟
        queryApiAndUpdate(LocalDateTime.now().minusMinutes(3),
                // 结束时间 现在
                LocalDateTime.now(),
                // 查询常规订单
                OrderTypeEnum.GENERAL_ORDER);
    }

    @Override
    public void syncChannelOrderStatus() {
        // 定时开关
        if (!orderTimerConfigProperties.getOrderStatusSyncQuartzFlag()) {
            log.info("定时开关未开启!");
            return;
        }

        // 渠道订单
        doSynchronizedIn30(OrderTypeEnum.CHANNEL_ORDER);
        doSynchronizedOver30(OrderTypeEnum.CHANNEL_ORDER);
    }

    @Override
    public void syncGeneralOrderStatus() {
        //定时开关
        if (!orderTimerConfigProperties.getPullOrderSwitchGeneral()) {
            log.info("定时开关未开启!");
            return;
        }
        doSynchronizedIn30(OrderTypeEnum.GENERAL_ORDER);
        doSynchronizedOver30(OrderTypeEnum.GENERAL_ORDER);
    }


    /**
     * 根据开始时间和结束时间，一个小时一个小时得查询接口并更新
     * @param start 开始时间
     * @param end 结束时间
     * @param type 订单类型
     */
    private void queryApiAndUpdateByDays(LocalDateTime start, LocalDateTime end, OrderTypeEnum type) {
        // 一个小时一个小时的同步
        while (start.isBefore(end)) {
            queryApiAndUpdate(start, (start = start.plusHours(1)), type);
        }
    }

    @Override
    public void queryApiAndUpdate(LocalDateTime start, LocalDateTime end, OrderTypeEnum type) {
        TbkOrderPageInfo<TbkOrderInfo> tbkOrderPageInfo;
        long pageNo = 1L;
        String positionIndex = null;

        do {

            log.info("查询订单信息: start :{} end: {}, pageNo: {} positionIndex: {} type: {}",
                    start, end, pageNo, positionIndex, type);

            tbkOrderPageInfo = tbkOrderApi.pullOrderNew(pageNo,
                    100L, positionIndex, start, end, type.getCode());

            // 无数据 终止循环
            if (Objects.isNull(tbkOrderPageInfo)) {
                break;
            }

            // 有数据 则处理
            if (!CollectionUtils.isEmpty(tbkOrderPageInfo.getList())) {
                tbkOrderPageInfo.getList().stream()
                        // 过滤出当前系统支持的site id
                        .filter(p -> Objects.equals(p.getSiteId(), tbkOpenOrderProperties.getBusinessSiteId()))
                        .forEach(this::handleOrderInfo);
            }

            pageNo = tbkOrderPageInfo.nextPageNo();
            positionIndex = tbkOrderPageInfo.getPositionIndex();

        } while (Objects.equals(tbkOrderPageInfo.getHasNext(), Boolean.TRUE));
    }

    /**
     * 同步1个月内的订单
     * @param type type
     */
    private void doSynchronizedIn30(OrderTypeEnum type) {
        LocalDateTime end = LocalDateTime.now();
        LocalDateTime start = end.minusMonths(1);
        queryApiAndUpdateByDays(start, end, type);
    }

    /**
     * 查询并处理30天以前的订单信息
     * @param type 订单类型
     */
    private void doSynchronizedOver30(OrderTypeEnum type) {
        // 同步1个月以前的未结算订单 避免时间跨度太长，这里直接进行查询，然后一个一个地进行处理
        List<LongTimeNoCheckOrderInfo> longTimeNoCheckOrders = getLongTimeNoCheckOrders(type);

        longTimeNoCheckOrders.forEach(p -> {
            log.info("处理历史订单: {} 创建时间: {}", p.getOrderSn(), p.getSourceOrderCreateTime());

            if (Objects.nonNull(p.getSourceOrderCreateTime())) {
                LocalDateTime end = p.getSourceOrderCreateTime().toInstant()
                        .atZone(ZoneId.systemDefault() )
                        .toLocalDateTime();

                // 前后都给2分钟
                queryApiAndUpdateByDays(end.minusMinutes(2), end.plusMinutes(2), type);
            }
        });
    }


    /**
     * 处理订单信息,插入订单信息,增加佣金
     * @param orderInfo 订单源信息
     * @return 处理是否成功
     */
    private Boolean handleOrderInfo(TbkOrderInfo orderInfo) {
        if (Objects.isNull(orderInfo)) {
            log.warn("购物订单数据为null 无法处理");
            return false;
        }

        // 饿了么订单处理
        if (Objects.equals(orderInfo.getTbOrderType(), TbkOrderType.ELE.getType())) {
            return eleTakeOutOrderService.syncOrder(orderInfo);
        }

        // 购物订单处理
        return handleCommodityOrderInfo(orderInfo);
    }

    /**
     * 购物订单处理
     * @param orderInfo orderInfo
     * @return 处理是否成功
     */
    public boolean handleCommodityOrderInfo(TbkOrderInfo orderInfo) {
        try {
            BusinesSiteId siteId = BusinesSiteId.getById(orderInfo.getSiteId());

            log.info("处理购物订单{}, 推广位id: {} 来源: {} 订单数据: {}", orderInfo.getOrderSn(), orderInfo.getSiteId(),
                    Objects.isNull(siteId) ? "未定义" : siteId.getDes(), JSON.toJSONString(orderInfo));

            return doHandleCommodityOrderInfo(orderInfo);
        } catch(Exception e) {
            log.error("处理购物订单失败 订单数据: {}", JSON.toJSONString(orderInfo), e);

            return false;
        }
    }

    /**
     * 获取30天以上的库内未结算订单进行查询二次更新
     * @param type 订单类型
     * @return 订单数据
     */
    protected abstract List<LongTimeNoCheckOrderInfo> getLongTimeNoCheckOrders(OrderTypeEnum type);

    /**
     * 处理商品订单信息
     * 方法内的处理流程为
     * 1. 判断订单是否存在
     *  1.1 存在
     *      1.1.1 存在则判断状态是否更新 未更新则跳过
     *      1.1.2 存在则判断状态是否更新 有更新更新状态
     *  1.2 不存在
     *      1.2.1 订单信息入库
     * @param orderInfo 订单信息
     * @return 处理是否成功
     */
    protected abstract boolean doHandleCommodityOrderInfo(TbkOrderInfo orderInfo);

}

