package com.bxm.newidea.component.config;

import org.springframework.aop.framework.autoproxy.BeanNameAutoProxyCreator;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.interceptor.TransactionInterceptor;

import javax.sql.DataSource;
import java.util.Properties;

/**
 * 服务层事务拦截策略
 */
@EnableTransactionManagement
@Configuration
@ConditionalOnMissingBean(name = "transactionInterceptor")
@ConditionalOnBean(DataSource.class)
public class TransactionConfiguration {

    @Bean(name = "transactionInterceptor")
    public TransactionInterceptor transactionInterceptor(DataSourceTransactionManager platformTransactionManager) {
        TransactionInterceptor transactionInterceptor = new TransactionInterceptor();
        // 事物管理器
        transactionInterceptor.setTransactionManager(platformTransactionManager);
        Properties transactionAttributes = new Properties();

        //必须包含事务的方法，如果使用到事务而不匹配设置的规则会发生异常（匹配规则）
        String[] requiredMethod = {"insert*",
                "new*",
                "set*",
                "add*",
                "save*",
                "create*",
                "push*",
                "send*",
                "update*",
                "write*",
                "modify*",
                "execute*",
                "excute*",
                "do*",
                "reg*",
                "process*",
                "bind*",
                "pull*",
                "batch*",
                "push*",
                "reset*",
                "cosume*",
                "lock*",
                "unlock*",
                "comple*",
                "change*",
                "confirm*",
                "pass*",
                "commit*",
                "abandon*",
                "del*",
                "call*",
                "remove*",
                "cancel*",
                "destroy*",
                "shift*",
                "apply*",
                "enable*",
                "disable*",
                "production*",
                "trans*",
                "waitOrder*"};
        //默认没有事务，如果调用方有事务则支持
        String[] supportMethod = {"get*", "load*", "query*", "find*", "is*", "obtain*", "read*", "exist*"};

        String required = "PROPAGATION_REQUIRED,-Throwable";
        String supports = "PROPAGATION_SUPPORTS,readOnly";

        for (String method : requiredMethod) {
            transactionAttributes.setProperty(method, required);
        }

        for (String method : supportMethod) {
            transactionAttributes.setProperty(method, supports);
        }

        transactionInterceptor.setTransactionAttributes(transactionAttributes);
        return transactionInterceptor;
    }

    @Bean
    public BeanNameAutoProxyCreator transactionAutoProxy() {
        BeanNameAutoProxyCreator transactionAutoProxy = new BeanNameAutoProxyCreator();
        //对非接口也进行代理
        transactionAutoProxy.setProxyTargetClass(true);
        //拦截规则
        transactionAutoProxy.setBeanNames("*ServiceImpl");
        //使用到的拦截器
        transactionAutoProxy.setInterceptorNames("transactionInterceptor");
        return transactionAutoProxy;
    }
}
