package com.bxm.newidea.component.vo;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.springframework.http.HttpStatus;

/**
 * 输出的标准Json格式数据， 用于替代原本黄瓜头条里面的Object输出
 * <code>ResultModel.build().data("mydata").code(Response.OK)</code>
 * @param <T> 泛型，Json中包含的具体的数据格式
 * @author liujia
 */
@ApiModel(description = "标准的Json响应格式数据")
public class ResultModel<T> {

    @ApiModelProperty(required = true, value = "操作结果编码,使用标准的Http status", example = "200,400,401,403,500等")
    private int code;

    @ApiModelProperty(value = "请求的返回消息，code为0可能为空，其他code均有值", example = "操作完成/操作成功/余额不足...")
    private String errorMsg = StringUtils.EMPTY;

    @ApiModelProperty(value = "请求的返回数据，根据请求不同有具体的返回实体")
    private T result;

    protected ResultModel() {
        this.code = HttpStatus.OK.value();
    }

    protected ResultModel(int code, String errorMsg) {
        this.code = code;
        this.errorMsg = errorMsg;
    }

    protected ResultModel(int code, String errorMsg, T result) {
        this.code = code;
        this.errorMsg = errorMsg;
        this.result = result;
    }

    protected ResultModel(ResultModel<T> source) {
        this.code = source.code;
        this.errorMsg = source.errorMsg;
        this.result = source.result;
    }

    public static <T> ResultModel build() {
        return new ResultModel();
    }

    public static <T> ResultModel build(T result) {
        ResultModel<T> resultModel = new ResultModel();
        return resultModel.setResult(result);
    }

    public static <T> ResultModel build(int code, String errorMsg) {
        return new ResultModel(code, errorMsg);
    }

    public static <T> ResultModel build(int code, String errorMsg, T result) {
        return new ResultModel<>(code, errorMsg, result);
    }

    /**
     * 正常返回的返回值模型
     */
    public static ResultModel ok() {
        return new ResultModel();
    }

    /**
     * 请求参数错误的返回模型
     */
    public static ResultModel badRequest() {
        return new ResultModel().setCode(HttpStatus.BAD_REQUEST.value());
    }

    /**
     * 服务器错误的返回模型
     */
    public static ResultModel serverError() {
        return new ResultModel().setCode(HttpStatus.INTERNAL_SERVER_ERROR.value());
    }

    /**
     * 操作是否成功
     * @return true表示操作成功
     */
    public boolean success() {
        return this.code == HttpStatus.OK.value();
    }

    public int getCode() {
        return this.code;
    }

    public ResultModel<T> setCode(int code) {
        this.code = code;
        return this;
    }

    public String getErrorMsg() {
        return this.errorMsg;
    }

    public ResultModel<T> setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public T getResult() {
        return this.result;
    }

    public ResultModel<T> setResult(T result) {
        this.result = result;
        return this;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
    }
}
