package com.bxm.newidea.component.controller;

import com.bxm.newidea.component.binder.ByteEditor;
import com.bxm.newidea.component.binder.DateEditor;
import com.bxm.newidea.component.tools.ValidBuilder;
import org.apache.commons.lang.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.TimeZone;

/**
 * controller 基类
 * 预留，后续用于处理公共内容
 * @author liujia 2018/3/30 17:20
 */
public class BaseController {

    protected static final String RESULT_KEY = "result";

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    private static String encodeURIComponent(String value) throws UnsupportedEncodingException {
        return URLEncoder.encode(value, "UTF-8").replaceAll("\\+", "%20");
    }

    /**
     * 属性值绑定转换编辑器
     * @param binder 绑定集合
     */
    @InitBinder
    public void initBinding(WebDataBinder binder) {
        //byte转换
        binder.registerCustomEditor(Byte.class, new ByteEditor());
        //string类型转换为空时，转换为null而不是空字符串
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(true));
        //使用自定义的日期转换
        binder.registerCustomEditor(Date.class, new DateEditor());
    }

    protected void write(HttpServletResponse response, String content) {
        try {
            response.getWriter().write(content);
            response.getWriter().flush();
        } catch (IOException e) {
            this.logger.error(e.getMessage(), e);
        }
    }

    protected void processDownload(String fileName, HttpServletResponse response) {
        try {
            String dateString = DateFormatUtils.format(new Date(), "EEE, dd MMM yyyy HH:mm:ss z", TimeZone.getTimeZone("GMT"));
            String contentType = "application/x-download";

            response.setStatus(HttpServletResponse.SC_OK);
            response.setHeader("Last-modified", dateString);
            response.setHeader("Connection", "keep-alive");
            response.setHeader("Date", dateString);
            response.setHeader("Cache-Control", "no-cache");
            response.setContentType(contentType);

            String headerValue = "attachment;";
            headerValue += " filename=\"" + encodeURIComponent(fileName) + "\";";
            headerValue += " filename*=utf-8''" + encodeURIComponent(fileName);
            response.setHeader("Content-Disposition", headerValue);
        } catch (UnsupportedEncodingException e) {
            this.logger.error("文件名转换发生异常,文件名为[{0}]", fileName);
        }
    }

    /**
     * 多个字段验证并联，从这个方法开始进行调用
     * eg：
     * combo().prop(filedName,fieldValue).notBlank().min(10).valid();
     * @return 验证构造器，需要最终调用<code>valid()</code>方法进行验证
     */
    protected ValidBuilder combo() {
        return new ValidBuilder(false);
    }

    /**
     * 单个属性进行校验时调用，如果后续的验证出错则直接抛出验证异常，不需要额外调用<code>valid()</code>
     * eg:
     * direct().prop(fieldName,fieldValue).notBlank();
     * @return 验证构造器，用于进行字段校验
     */
    protected ValidBuilder direct(){
        return new ValidBuilder(true);
    }
}
