package com.bxm.newidea.component.schedule.config;

import com.bxm.component.graceful.shutdown.event.ShutdownEvent;
import com.bxm.component.httpclient.utils.NetworkInterfaceManager;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.rpc.ScheduleFeignService;
import com.bxm.newidea.component.schedule.rpc.ScheduleServiceImpl;
import com.bxm.newidea.component.schedule.rpc.impl.DefaultScheduleFeignService;
import com.bxm.newidea.component.schedule.rpc.impl.ScheduleFeignServiceImpl;
import com.bxm.newidea.component.schedule.task.ScheduleTask;
import com.xxl.job.core.executor.impl.XxlJobSpringExecutor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.util.Map;

/**
 * xxl联动的调度服务启动配置
 * @author liujia
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(XxlJobConfigurationProperties.class)
@ConditionalOnMissingClass("org.springframework.test.context.junit4.SpringRunner")
public class XxlJobConfiguration {
    private static final String EXECUTOR_IP = "executorIp";

    private final XxlJobConfigurationProperties properties;

    public XxlJobConfiguration(XxlJobConfigurationProperties properties) {
        this.properties = properties;
    }

    @Bean
    public ScheduleFeignService feignService() {
        if (properties.isMock()) {
            log.info("使用默认实现的ScheduleFeignService");
            return new DefaultScheduleFeignService();
        }

        log.info("使用http实现的ScheduleFeignService");
        return new ScheduleFeignServiceImpl(properties);
    }

    @Bean
    public ScheduleService scheduleService(ScheduleFeignService feignService) {
        return new ScheduleServiceImpl(properties, feignService);
    }

    @Bean
    public BeanByNameHandler beanByNameHandler() {
        return new BeanByNameHandler();
    }

    @Bean(initMethod = "start", destroyMethod = "destroy")
    @ConditionalOnProperty(name = "xxl.job.executor.app-name")
    public XxlJobSpringExecutor xxlJobExecutor() {
        XxlJobSpringExecutor xxlJobSpringExecutor = new XxlJobSpringExecutor();

        if (StringUtils.isBlank(properties.getAppName())) {
            throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.app-name配置属性");
        }

        if (StringUtils.isBlank(properties.getIp())) {
            String ip = System.getProperty(EXECUTOR_IP);
            if (StringUtils.isBlank(ip)) {
                ip = NetworkInterfaceManager.INSTANCE.getLocalHostAddress();
                if (null == ip) {
                    throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.ip配置属性");
                }
            }
            xxlJobSpringExecutor.setIp(ip);
        }

        if (StringUtils.isBlank(properties.getAdminAddresses())) {
            throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.admin-address配置属性");
        }

        if (XxlJobConfigurationProperties.DEFAULT_PORT == properties.getPort()) {
            log.info("调度服务回调接口使用了默认的[{}]端口，请注意部署时的端口冲突问题", properties.getPort());
        }

        xxlJobSpringExecutor.setAdminAddresses(properties.getAdminAddresses());
        xxlJobSpringExecutor.setAppName(properties.getAppName());
        xxlJobSpringExecutor.setPort(properties.getPort());
        xxlJobSpringExecutor.setAccessToken(properties.getAccessToken());
        xxlJobSpringExecutor.setLogPath(properties.getLogPath());
        xxlJobSpringExecutor.setLogRetentionDays(properties.getLogRetentionDays());

        return xxlJobSpringExecutor;
    }

    @Order(Ordered.HIGHEST_PRECEDENCE)
    @EventListener(ApplicationStartedEvent.class)
    public void initTask(ApplicationStartedEvent event) {
        //服务启动完毕后开始创建
        ScheduleService scheduleService = event.getApplicationContext().getBean(ScheduleService.class);
        //获取spring中配置的调度任务
        Map<String, ScheduleTask> taskMap = event.getApplicationContext().getBeansOfType(ScheduleTask.class);

        log.info("初始化定时任务，任务数量：[{}]", taskMap.size());

        if (taskMap.size() > 0) {
            taskMap.values().forEach(task -> {
                log.info("初始化定时任务，任务名称：[{}],任务描述：[{}]", task.taskName(), task.description());
                scheduleService.push(task);
            });
        }
    }

    /**
     * 响应自定义的shutdown event
     * 将执行器服务下线，并阻塞一段时间，避免客户端缓存
     * @param event 监听事件
     */
    @Order(Ordered.HIGHEST_PRECEDENCE)
    @EventListener(ShutdownEvent.class)
    public void onApplicationEvent(ShutdownEvent event) {
        xxlJobExecutor().destroy();
    }

}