package com.bxm.newidea.component.schedule.rpc;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.config.ExecutorParam;
import com.bxm.newidea.component.schedule.config.XxlJobConfigurationProperties;
import com.bxm.newidea.component.schedule.constant.CommonConstant;
import com.bxm.newidea.component.schedule.task.ScheduleTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;

/**
 * 使用配置中的远程调度服务地址，通过http请求创建、更新、删除定时任务
 * @author liujia
 */
@Slf4j
public class ScheduleServiceImpl implements ScheduleService {

    private ScheduleFeignService scheduleFeignService;

    private XxlJobConfigurationProperties properties;

    private static final String VERSION_SHORT_KEY = "v";

    @Value("${eureka.instance.metadataMap.version:}")
    private String serviceVersion;

    public ScheduleServiceImpl(XxlJobConfigurationProperties properties,
                               ScheduleFeignService scheduleFeignService) {
        this.properties = properties;
        this.scheduleFeignService = scheduleFeignService;
    }

    @Override
    public boolean push(ScheduleTask task) {
        if (validate(task)) {
            //临时性任务要做更新
            if (task.onceTask()) {
                scheduleFeignService.update(buildTaskName(task),
                        task.cron(),
                        task.description(),
                        properties.getAppName(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            } else {
                //永久性任务只做新增
                scheduleFeignService.add(buildTaskName(task),
                        task.cron(),
                        task.description(),
                        properties.getAppName(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            }
            return true;
        }
        return false;
    }

    /**
     * 在定时任务后追加版本号标记,便于控制不同版本的定时任务
     * 任务中心使用taskName作为唯一标记，通过版本区分后，该任务对应回调地址就会与微服务一致，用于解决相同微服务版本不同时的回调问题
     * @param task 定时任务对象
     * @return 重构后的任务名称
     */
    private String buildTaskName(ScheduleTask task) {
        //任务本身的版本配置
        String version = task.version();

        //配置中心中指定的调度中心版本
        if (StringUtils.isBlank(version)) {
            version = properties.getVersion();
        }

        //尝试获取当前服务的注册元数据中的版本号
        if (StringUtils.isBlank(version)) {
            version = serviceVersion;
        }

        //系统属性中指定的配置
        if (StringUtils.isBlank(version)) {
            version = System.getProperty("version");
        }

        if (StringUtils.isNotBlank(version) && !StringUtils.startsWithIgnoreCase(version, VERSION_SHORT_KEY)) {
            if (StringUtils.startsWithIgnoreCase(version, VERSION_SHORT_KEY)) {
                version = "_" + version;
            } else {
                version = "_" + VERSION_SHORT_KEY + version;
            }
        }

        if (StringUtils.isBlank(version)) {
            return task.taskName();
        } else {
            return task.taskName() + version;
        }
    }

    @Override
    public boolean remove(String taskName) {
        if (StringUtils.isNotBlank(taskName)) {
            scheduleFeignService.remove(taskName);
            return true;
        }
        return false;
    }

    private String buildParam(ScheduleTask task) {
        return JSON.toJSONString(new ExecutorParam(task.callback().beanName(), task.callbackParam()));
    }

    private boolean validate(ScheduleTask task) {
        if (StringUtils.isBlank(task.taskName())) {
            log.error("任务名未配置");
            return false;
        }
        if (null == task.cron()) {
            log.error("任务未设置执行时间");
            return false;
        }
        if (null == task.callback()) {
            log.error("任务回调服务类未设置");
            return false;
        }

        return true;
    }
}

