package com.bxm.newidea.component.schedule.config;

import com.bxm.newidea.component.schedule.OnceJobExecutor;
import com.bxm.newidea.component.tools.NetworkInterfaceManager;
import com.xxl.job.core.executor.XxlJobExecutor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

/**
 * xxl联动的调度服务启动配置
 *
 * @author liujia
 */
@Slf4j
@Configuration
@AutoConfigureOrder(Ordered.LOWEST_PRECEDENCE)
@EnableConfigurationProperties(XxlJobConfigurationProperties.class)
public class XxlJobConfiguration {

    private final XxlJobConfigurationProperties properties;

    private static final String EXECUTOR_IP = "executorIp";

    public XxlJobConfiguration(XxlJobConfigurationProperties properties) {
        this.properties = properties;
    }

    @Bean(destroyMethod = "destroy")
    public XxlJobExecutor xxlJobExecutor() {
        ComponentXxlJobExecutor xxlJobExecutor = new ComponentXxlJobExecutor();

        if (properties.isMock()) {
            log.info("当前使用模拟定时任务执行器，不进行初始化");
            return xxlJobExecutor;
        }

        if (StringUtils.isBlank(properties.getAppName())) {
            throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.app-name配置属性");
        }

        // 优先获取环境变量中配置的IP
        String ip = System.getProperty(EXECUTOR_IP);
        if (StringUtils.isBlank(ip)) {
            // 尝试获取本机IP
            ip = NetworkInterfaceManager.INSTANCE.getLocalHostAddress();
            if (null == ip) {
                throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.ip配置属性");
            }
            xxlJobExecutor.setIp(ip);
        }

        if (StringUtils.isBlank(properties.getAdminAddresses())) {
            throw new IllegalArgumentException("使用调度服务，必须提供xxl.job.executor.adminAddresses配置属性");
        }

        if (XxlJobConfigurationProperties.DEFAULT_PORT == properties.getPort()) {
            log.info("调度服务回调接口使用了默认的[{}]端口，请注意部署时的端口冲突问题", properties.getPort());
        }
        int port = properties.getPort();
        if (properties.getPort() <= 0) {
            port = properties.getAppPort() + 1000;
            log.info("未配置暴露的xxl-job服务端口，使用：{}", port);
        }

        xxlJobExecutor.setAdminAddresses(properties.getAdminAddresses());
        xxlJobExecutor.setAppname(properties.getAppName());
        xxlJobExecutor.setPort(port);
        xxlJobExecutor.setAccessToken(properties.getAccessToken());
        xxlJobExecutor.setLogPath(properties.getLogPath());
        xxlJobExecutor.setLogRetentionDays(properties.getLogRetentionDays());

        xxlJobExecutor.setXxlJobExecutorExtend(xxlJobExecutorExtend());

        try {
            xxlJobExecutor.start();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return xxlJobExecutor;
    }

    @Bean
    public OnceJobExecutor onceJobExecutor(XxlJobExecutorExtend xxlJobExecutorExtend) {
        return new OnceJobExecutor(properties, xxlJobExecutorExtend);
    }

    @Bean
    public XxlJobExecutorExtend xxlJobExecutorExtend() {
        return new XxlJobExecutorExtend(properties);
    }
}