package com.bxm.newidea.component.zk.service.impl;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.zk.exception.ZookeeperException;
import com.bxm.newidea.component.zk.listener.NodeChangeListener;
import com.bxm.newidea.component.zk.listener.NodeChangeWithChildrenListener;
import com.bxm.newidea.component.zk.service.ZookeeperAdapter;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.NodeCache;
import org.apache.curator.framework.recipes.cache.NodeCacheListener;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 6/7/21 9:23 AM
 **/
@Slf4j
public class ZookeeperAdapterImpl implements ZookeeperAdapter {

    private CuratorFramework curatorFramework;

    private ThreadLocal<CuratorFramework> curatorFrameworkThreadLocal;

    private static final String JOIN_CHAR = "/";

    public ZookeeperAdapterImpl(CuratorFramework curatorFramework) {
        this.curatorFramework = curatorFramework;
    }

    @Override
    public void changeCuratorFramework(CuratorFramework curatorFramework) {
        curatorFrameworkThreadLocal = new ThreadLocal<>();
        curatorFrameworkThreadLocal.set(curatorFramework);
    }

    @Override
    public String createNode(String path) {
        return createNode(path, CreateMode.PERSISTENT, null);
    }

    @Override
    public <T> String createNode(String path, T data) {
        return createNode(path, CreateMode.PERSISTENT, data);
    }

    @Override
    public String createNode(String path, CreateMode createMode) {
        return createNode(path, createMode, null);
    }

    @Override
    public <T> String createNode(String path, CreateMode createMode, T data) {
        checkPath(path);

        String nodePath = null;
        try {
            nodePath = getCuratorFramework().create()
                    .creatingParentsIfNeeded()
                    .withMode(createMode)
                    .forPath(path, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return nodePath;
    }

    @Override
    public void removeNode(String path) {
        checkPath(path);

        try {
            getCuratorFramework().delete().forPath(path);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    @Override
    public void removeNodeWithChildren(String path) {
        checkPath(path);

        try {
            if (log.isDebugEnabled()) {
                log.debug("同时删除[{}]下的子节点，请确定是否业务需要如此", path);
            }

            getCuratorFramework().delete().deletingChildrenIfNeeded().forPath(path);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    @Override
    public <T> Stat setData(String path, T data) {
        checkPath(path);

        Stat stat = null;
        try {
            stat = getCuratorFramework()
                    .setData()
                    .forPath(path, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public <T> Stat setData(String path, T data, int version) {
        checkPath(path);

        Stat stat = null;
        try {
            stat = getCuratorFramework()
                    .setData()
                    .withVersion(version)
                    .forPath(path, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public List<String> obtainChildren(String path, boolean fullPath) {
        checkPath(path);

        List<String> subPathList = Lists.newArrayList();
        try {
            subPathList = getCuratorFramework().getChildren().forPath(path);

            if (fullPath) {
                subPathList = subPathList.stream().map(subPath -> {
                    if (path.endsWith(JOIN_CHAR)) {
                        return path + subPath;
                    }
                    return path + JOIN_CHAR + subPath;
                }).collect(Collectors.toList());
            }
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }

        return subPathList;
    }

    @Override
    public <T> T obtainData(String path, Class<T> dataClass) {
        checkPath(path);

        T result = null;
        try {
            if (null == dataClass) {
                log.error("调用了[obtainData]但是未提供dataClass参数，直接返回null");
                return null;
            }

            byte[] sourceData = getCuratorFramework().getData().forPath(path);

            result = restoreData(sourceData, dataClass);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }


        return result;
    }

    @Override
    public Stat obtainStat(String path) {
        checkPath(path);

        Stat stat = null;

        try {
            stat = getCuratorFramework().checkExists().forPath(path);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public boolean checkExists(String path) {
        return obtainStat(path) != null;
    }

    @Override
    public void addListener(String path, NodeChangeListener listener) {
        checkPath(path);

        NodeCache nodeCache = new NodeCache(getCuratorFramework(), path, false);

        nodeCache.getListenable().addListener(new NodeCacheListener() {
            @Override
            public void nodeChanged() throws Exception {
                ChildData currentData = nodeCache.getCurrentData();
                currentData.getData();
                //TODO 泛型适配
            }
        });

    }

    @Override
    public void addListener(String path, NodeChangeWithChildrenListener listener, boolean self) {
        //TODO
    }

    private void warpException(Throwable throwable) throws ZookeeperException {
        log.error(throwable.getMessage(), throwable);
        throw new ZookeeperException(throwable);
    }

    private void checkPath(String path) {
        if (StringUtils.isEmpty(path)) {
            throw new ZookeeperException("path必须提供");
        }
    }

    /**
     * 将节点值转化为json进行存储
     *
     * @param source 原始数据
     * @return 原始数据转化为json字符串后的byte数组
     */
    private <T> byte[] toData(T source) {
        if (null == source) {
            return getDefaultData();
        }

        if (byte[].class.equals(source.getClass())) {
            return (byte[]) source;
        }

        String jsonString = JSON.toJSONString(source);

        if (null == jsonString) {
            return source.toString().getBytes();
        }

        return jsonString.getBytes();
    }

    private <T> T restoreData(byte[] source, Class<T> dataClass) {
        String sourceString = new String(source);
        return JSON.parseObject(sourceString, dataClass);
    }

    /**
     * 节点未提供设置的值时，写入默认值
     *
     * @return 默认写入节点的创建时间
     */
    private byte[] getDefaultData() {
        return (System.currentTimeMillis() + "").getBytes();
    }

    private CuratorFramework getCuratorFramework() {
        if (null != curatorFrameworkThreadLocal) {
            CuratorFramework localFramework = curatorFrameworkThreadLocal.get();
            if (null != localFramework) {
                return localFramework;
            }
        }

        return curatorFramework;
    }

    private void release() {
        if (null != curatorFrameworkThreadLocal) {
            curatorFrameworkThreadLocal.remove();
        }
    }
}
