package com.bxm.newidea.component.zk.service.impl;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.zk.exception.ZookeeperException;
import com.bxm.newidea.component.zk.listener.NodeChangeListener;
import com.bxm.newidea.component.zk.listener.NodeChangeWithChildrenListener;
import com.bxm.newidea.component.zk.service.ZookeeperAdapter;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.*;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 6/7/21 9:23 AM
 **/
@Slf4j
public class ZookeeperAdapterImpl implements ZookeeperAdapter {

    private CuratorFramework curatorFramework;

    private ThreadLocal<CuratorFramework> curatorFrameworkThreadLocal;

    private static final String JOIN_CHAR = "/";

    public ZookeeperAdapterImpl(CuratorFramework curatorFramework) {
        this.curatorFramework = curatorFramework;
    }

    @Override
    public void changeCuratorFramework(CuratorFramework curatorFramework) {
        curatorFrameworkThreadLocal = new ThreadLocal<>();
        curatorFrameworkThreadLocal.set(curatorFramework);
    }

    @Override
    public String createNode(String fullPath) {
        return createNode(fullPath, CreateMode.PERSISTENT, null);
    }

    @Override
    public <T> String createNode(String fullPath, T data) {
        return createNode(fullPath, CreateMode.PERSISTENT, data);
    }

    @Override
    public String createNode(String fullPath, CreateMode createMode) {
        return createNode(fullPath, createMode, null);
    }

    @Override
    public <T> String createNode(String fullPath, CreateMode createMode, T data) {
        checkPath(fullPath);

        String nodePath = null;
        try {
            nodePath = getCuratorFramework().create()
                    .creatingParentsIfNeeded()
                    .withMode(createMode)
                    .forPath(fullPath, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return nodePath;
    }

    @Override
    public void removeNode(String fullPath) {
        checkPath(fullPath);

        try {
            getCuratorFramework().delete().forPath(fullPath);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    @Override
    public void removeNodeWithChildren(String fullPath) {
        checkPath(fullPath);

        try {
            if (log.isDebugEnabled()) {
                log.debug("同时删除[{}]下的子节点，请确定是否业务需要如此", fullPath);
            }

            getCuratorFramework().delete().deletingChildrenIfNeeded().forPath(fullPath);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    @Override
    public <T> Stat setData(String fullPath, T data) {
        checkPath(fullPath);

        Stat stat = null;
        try {
            stat = getCuratorFramework()
                    .setData()
                    .forPath(fullPath, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public <T> Stat setData(String fullPath, T data, int version) {
        checkPath(fullPath);

        Stat stat = null;
        try {
            stat = getCuratorFramework()
                    .setData()
                    .withVersion(version)
                    .forPath(fullPath, toData(data));
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public List<String> obtainChildren(String fullPath, boolean fillFullPath) {
        checkPath(fullPath);

        List<String> subPathList = Lists.newArrayList();
        try {
            subPathList = getCuratorFramework().getChildren().forPath(fullPath);

            if (fillFullPath) {
                subPathList = subPathList.stream().map(subPath -> {
                    if (fullPath.endsWith(JOIN_CHAR)) {
                        return fullPath + subPath;
                    }
                    return fullPath + JOIN_CHAR + subPath;
                }).collect(Collectors.toList());
            }
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }

        return subPathList;
    }

    @Override
    public <T> T obtainData(String fullPath, Class<T> dataClass) {
        checkPath(fullPath);

        T result = null;
        try {
            if (null == dataClass) {
                log.error("调用了[obtainData]但是未提供dataClass参数，直接返回null");
                return null;
            }

            byte[] sourceData = getCuratorFramework().getData().forPath(fullPath);

            result = restoreData(sourceData, dataClass);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }


        return result;
    }

    @Override
    public Stat obtainStat(String fullPath) {
        checkPath(fullPath);

        Stat stat = null;

        try {
            stat = getCuratorFramework().checkExists().forPath(fullPath);
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
        return stat;
    }

    @Override
    public boolean checkExists(String fullPath) {
        return obtainStat(fullPath) != null;
    }

    @Override
    public void addListener(String fullPath, NodeChangeListener listener) {
        checkPath(fullPath);
        Preconditions.checkArgument(null != listener);

        log.info("添加[{}]监听器", fullPath);

        try {
            NodeCache nodeCache = new NodeCache(getCuratorFramework(), fullPath, false);

            nodeCache.getListenable().addListener(() -> {
                ChildData currentData = nodeCache.getCurrentData();
                if (null == currentData) {
                    log.info("[{}]监听事件回调值为空", fullPath);
                    return;
                }

                listener.updated(fullPath, restoreData(currentData.getData()), currentData.getStat());
            });

            nodeCache.start();
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    @Override
    public void addListener(String fullPath, NodeChangeWithChildrenListener listener, boolean self) {
        checkPath(fullPath);
        Preconditions.checkArgument(null != listener);

        log.info("添加[{}]子节点监听器，是否监听自身:[{}]", fullPath, self);

        try {
            if (self) {
                addTreeNodeCache(fullPath, listener);
            } else {
                addPathChildrenCache(fullPath, listener);
            }
        } catch (Exception e) {
            warpException(e);
        } finally {
            release();
        }
    }

    private void addPathChildrenCache(String path, NodeChangeWithChildrenListener listener) throws Exception {
        PathChildrenCache pathChildrenCache = new PathChildrenCache(getCuratorFramework(), path, false);

        pathChildrenCache.getListenable().addListener((client, event) -> {
            ChildData childData = event.getData();
            if (null == childData) {
                log.info("[{}]监听事件回调值为空", path);
                return;
            }

            String data = restoreData(childData.getData());
            PathChildrenCacheEvent.Type type = event.getType();

            switch (type) {
                case CONNECTION_RECONNECTED:
                    pathChildrenCache.rebuild();
                    break;
                case CHILD_ADDED:
                    listener.added(childData.getPath(), data, childData.getStat());
                    break;
                case CHILD_REMOVED:
                    listener.removed(childData.getPath(), data, childData.getStat());
                    break;
                case CHILD_UPDATED:
                    listener.updated(childData.getPath(), data, childData.getStat());
                    break;
                default:
                    break;
            }
        });

        pathChildrenCache.start();
    }

    private void addTreeNodeCache(String path, NodeChangeWithChildrenListener listener) throws Exception {
        TreeCache treeCache = new TreeCache(getCuratorFramework(), path);

        treeCache.getListenable().addListener((client, event) -> {
            ChildData childData = event.getData();
            if (null == childData) {
                log.info("[{}]监听事件回调值为空", path);
                return;
            }

            TreeCacheEvent.Type type = event.getType();
            String data = restoreData(childData.getData());

            switch (type) {
                case NODE_ADDED:
                    listener.added(childData.getPath(), data, childData.getStat());
                    break;
                case NODE_REMOVED:
                    listener.removed(childData.getPath(), data, childData.getStat());
                    break;
                case NODE_UPDATED:
                    listener.updated(childData.getPath(), data, childData.getStat());
                    break;
                default:
                    break;
            }
        });

        treeCache.start();
    }

    private void warpException(Throwable throwable) throws ZookeeperException {
        log.error(throwable.getMessage(), throwable);
        throw new ZookeeperException(throwable);
    }

    private void checkPath(String path) {
        if (StringUtils.isEmpty(path)) {
            throw new ZookeeperException("path必须提供");
        }
    }

    /**
     * 将节点值转化为json进行存储
     *
     * @param source 原始数据
     * @return 原始数据转化为json字符串后的byte数组
     */
    private <T> byte[] toData(T source) {
        if (null == source) {
            return getDefaultData();
        }

        if (byte[].class.equals(source.getClass())) {
            return (byte[]) source;
        }

        String jsonString = JSON.toJSONString(source);

        if (null == jsonString) {
            return source.toString().getBytes();
        }

        return jsonString.getBytes();
    }

    private String restoreData(byte[] source) {
        if (null == source) {
            return null;
        }

        return new String(source);
    }

    private <T> T restoreData(byte[] source, Class<T> dataClass) {
        String sourceString = new String(source);

        return JSON.parseObject(sourceString, dataClass);
    }

    /**
     * 节点未提供设置的值时，写入默认值
     *
     * @return 默认写入节点的创建时间
     */
    private byte[] getDefaultData() {
        return (System.currentTimeMillis() + "").getBytes();
    }

    private CuratorFramework getCuratorFramework() {
        if (null != curatorFrameworkThreadLocal) {
            CuratorFramework localFramework = curatorFrameworkThreadLocal.get();
            if (null != localFramework) {
                return localFramework;
            }
        }

        return curatorFramework;
    }

    private void release() {
        if (null != curatorFrameworkThreadLocal) {
            curatorFrameworkThreadLocal.remove();
        }
    }
}
